variable "template_name" {
  default = "user_data.tpl"
}

# invoke the module for tagging
module "taglib" {
  source      = "git::git+ssh://git@git.xarth.tv/terraform-modules/taglib?ref=v0.4"
  name        = "${var.role}-${var.environment}"
  owner       = var.owner
  service     = "${var.role}/${var.service}"
  role        = var.role
  srctag      = "aws-instances"
  tags        = local.instance_tags
  environment = var.environment
}

data "aws_iam_instance_profile" "aws_instance_profile" {
  name       = "${var.role}-${var.environment}"
  depends_on = [aws_iam_instance_profile.instance_profile]
}

# Basic IAM Role
resource "aws_iam_role" "iam_role" {
  name  = "${var.role}-${var.environment}"
  count = var.instance_profile == "create" ? 1 : 0
  path  = "/"
  tags  = module.taglib.tags

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "AmazonEC2RoleforSSM" {
  count      = var.instance_profile == "create" ? 1 : 0
  role       = aws_iam_role.iam_role[count.index].name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM"
}

# Instance profile for the instances
resource "aws_iam_instance_profile" "instance_profile" {
  name       = "${var.role}-${var.environment}"
  count      = var.instance_profile == "create" ? 1 : 0
  role       = aws_iam_role.iam_role[count.index].name
  depends_on = [aws_iam_role.iam_role]
}

# https://github.com/terraform-providers/terraform-provider-aws/issues/6086#issuecomment-443317835
data "aws_ami" "canonical_ami" {
  most_recent = true
  owners      = ["099720109477"]

  filter {
    name   = "name"
    values = [local.canonical_search]
  }

  filter {
    name   = "virtualization-type"
    values = ["hvm"]
  }
}

data "aws_region" "current" {}

# Load user_data
data "template_file" "user_data" {
  template = file(join("/", list(path.module, var.template_name)))

  vars = {
    files       = local.files
    facts       = module.taglib.ec2data_fact_stanza
    service     = var.service
    role        = var.role
    owner       = var.owner
    environment = var.environment
    puppet      = local.puppet_server
    dns_region  = data.aws_region.current.name
    dns_zone    = replace(data.aws_route53_zone.r53zone.name, "/\\.$/", "")
    run_scripts = var.run_scripts
    packages    = var.ami == "bionic" && local.puppet_server != "echo no puppet" ? "omnibus-puppet" : local.puppet_server == "echo no puppet" ? "" : "puppet"
  }
}

# calculate tags in a local for interpolation expression
# requires terraform 0.10.3 or newer
locals {
  /* static private IP vs basic invocation - THEORY
     unwind whether we are using private ip assignments or not.
      if we _are_...
       - require the "count" be 0.
       - require the "subnet_ids" list be empty.
      variables further explained inline.
   */

  /* pip_mode
     get the state of the private_ip var by checking if it's empty.
      1 - private ip is in play
      0 - use automatic-derivation
     this is used as a toggle for configuring the instance count.
   */

  pip_mode = length(var.private_ip)

  /* *_check
     the _check variables are used as assertions.
     if pip_mode is 0 (so, no static ips), leave them as zero. otherwise
      set them to the count or subnet list length as handed in. if the caller
      did not set those vars, they should _stay_ zero. otherwise we are about
      to have a bad time and should grumble.
   */

  count_check  = local.pip_mode == 0 ? 0 : var.instance_count
  subnet_check = local.pip_mode == 0 ? 0 : length(var.subnet_ids)

  /* instance_count
     the instance_count is switched based on derivation from the list of
      private ips handed in, or the count asked for.
   */

  instance_count = local.pip_mode == 0 ? var.instance_count : length(var.private_ip)

  /* subnet_ids
     this is either var.subnet_ids or computed from private_ip.
     combining the lists at this point is fine, as _one_ of them should be
      empty. conditionals are not allowed to return lists.
   */

  subnet_ids = concat(var.subnet_ids,keys(var.private_ip))

  /* puppet_server
    set puppet command if puppet or server is provided
    the default "echo" here is intended to specify no puppet command is run.
   */

  puppet_server_map = map(
    "puppet",                        "puppet agent --enable ; puppet agent --test --server=puppet.central.twitch.a2z.com --environment=master",
    "puppet.internal.justin.tv",     "puppet agent --enable ; puppet agent --test --server=puppet.central.twitch.a2z.com --environment=master",
    "puppet.central.twitch.a2z.com", "puppet agent --enable ; puppet agent --test --server=puppet.central.twitch.a2z.com --environment=master",
    "a2z",                           "puppet agent --enable ; puppet agent --test --server=puppet.systems.twitch.a2z.com --environment=master",
    "puppet.systems.twitch.a2z.com", "puppet agent --enable ; puppet agent --test --server=puppet.systems.twitch.a2z.com --environment=master"
  )
  puppet_server = lookup(local.puppet_server_map,"${var.puppet}","echo no puppet")

  /* ami config
    You may pass in your own ami, or "xenial" or "bionic" for public versions of those OSes.
   */

  ami_map = map(
    "xenial", "ubuntu/images/hvm-ssd/ubuntu-xenial-16.04-amd64-server-*",
    "bionic", "ubuntu/images/hvm-ssd/ubuntu-bionic-18.04-amd64-server-*",
    "focal",  "ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64-server-*",
  )
  canonical_ami    = lookup(local.ami_map,"${var.ami}","NONE")
  canonical_search = local.canonical_ami == "NONE" ? "ubuntu/images/hvm-ssd/ubuntu-*-amd64-server-*" : local.canonical_ami

  /* environment_tagname
     For use with tagging semantics as laid out by the capacity planning team
     Used with the Environment tag given to the EC2 instance
   */

  environment_tagname_map = map("production", "prod",
                                   "staging",    "stage",
				   "development","dev")
  environment_tagname = lookup(local.environment_tagname_map,var.environment)

  /* instance_tags
     this is calculated here due to using dynamic key names.
     the last two tags are for codedeploy. the repo used for deployment
      should == role of these instances
   */

  instance_tags = merge(var.additional_tags,map(
    "Role",        var.role,
    "App",         var.service,
    "Team",        var.owner,
    "Environment", local.environment_tagname
  ))
}

/*
   the below two null_resouces should never appear in your plan.
   they are alarm bells you have conflicting module parameters
   and, if you try to apply, will return apply failures. they will _not_
   stop the module from actually running though.
 */

# raise exception if we have a non-zero count *and* private IPs.
resource "null_resource" "count_check_alarm" {
  count = local.count_check

  provisioner "local-exec" {
    command = "false"
  }
}

# raise exception if we have subnets specified *and* private IPs.
resource "null_resource" "subnet_check_alarm" {
  count = local.subnet_check

  provisioner "local-exec" {
    command = "false"
  }
}

# AWS Instances
resource "aws_instance" "instance" {
  ami                    = local.canonical_ami == "NONE" ? var.ami : data.aws_ami.canonical_ami.id
  count                  = local.instance_count
  instance_type          = var.instance_type
  vpc_security_group_ids = var.vpc_security_group_ids

  # Stripe instances across the specified subnets
  subnet_id = element(local.subnet_ids, count.index % length(local.subnet_ids))

  root_block_device {
    volume_size = var.root_block_device_size
  }

  iam_instance_profile = var.instance_profile == "create" ? data.aws_iam_instance_profile.aws_instance_profile.name : var.instance_profile

  # Render and pass in the user_data template
  user_data = data.template_file.user_data.rendered

  tags = merge(module.taglib.tags,map(
    "instance-${var.owner}-${var.role}-${var.environment}","1",
    "ServiceTuple:${var.chasqui_service}:${var.chasqui_stage}","::"))

  depends_on = [aws_iam_instance_profile.instance_profile]

  private_ip = lookup(var.private_ip, element(local.subnet_ids, count.index % length(local.subnet_ids)),"")

  # ignore lifecycle _changes_. you want that? go delete the instance kthxbai
  lifecycle { ignore_changes = [user_data] }
}
