data "aws_caller_identity" "current" { }

output "account_id" {
  value = "${data.aws_caller_identity.current.account_id}"
}

variable "environment" {
}

variable "queue_region" {
}

variable "queue_account_number" {
}

variable "queue_name" {
}

variable "bucket_name" {
}

variable "node_count" {
    default = "1"
}

module "baseami" {
    source = "git::git+ssh://git@git-aws.internal.justin.tv/dta/tf_baseami_current?ref=1.0.0"
        
    datacenter = "${var.region}"
    ami_name = "baseami-xenial"
}

variable "ami" {
    description = "The AMI to use -- must be able to run apt-get install -y docker.io to install Docker"
    default = "default"
}

variable "image_version" {
    default = "latest"
}

variable "es_bulk_url" {}
variable "owner_email" {}
variable "owner" {}

variable "subnet_ids" {type = "list"}
variable "security_groups" {type = "list"}
variable "azs" {type = "list"}
variable "region" {
    default = "us-west-2"
}

variable "instance_type" {
    default = "r4.xlarge"
}

variable "es_bulk_kbytes" {
	default = "15360"
}
variable "es_bulk_size" {
	default = "300"
}
variable "es_message_builders" {
	default = "32"
}
variable "es_write_deadline" {
	default = "10s"
}
variable "es_writers" {
	default = "16"
}
variable "graphite_host" {
	default = "graphite.internal.justin.tv:2003"
}
variable "graphite_interval" {
	default = "10s"
}
variable "graphite_prefix" {
	default = "stats.counters.syslogv2.{{host}}.logbridge."
}
variable "s3_workers" {
	default = "4"
}
variable "sqs_workers" {
	default = "1"
}
variable "write_timeout" {
	default = "120s"
}

variable "extra_options" {
    default = ""
}

variable "syslog_host" {
    default = "grand-central-syslog.prod.us-west2.justin.tv"
}

variable "syslog_port" {
    default = "6514"
}

variable "syslog_tag" {
}

variable "syslog_facility" {
    default = "local0"
}

# These are for the puppet facts used to maintain the hosts.  Since systems
# maintains all normal hosts, these default to ones that make it possible for
# systems to push out changes to the whole cluster easily.
variable "puppet_team" {
    default = "systems"
}
variable "puppet_twitch_role" {
    default = "systems-central-elk-bridge"
}
variable "puppet_cluster" {
    default = "systems-central-elk"
}

resource "aws_iam_role" "iam_role" {
    name = "${var.bucket_name}-bridge"
    path = "/"
    assume_role_policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "sts:AssumeRole",
            "Principal": {
                "Service": "ec2.amazonaws.com"
            },
            "Effect": "Allow",
            "Sid": ""
        }
    ]
}
    EOF
    #<<EOF
}

resource "aws_iam_instance_profile" "instance_profile" {
    name = "${var.bucket_name}-bridge-profile"
    roles = ["${aws_iam_role.iam_role.id}"]
    depends_on = ["aws_iam_role.iam_role"]
}

resource "aws_iam_role_policy" "sqs_policy" {
  name = "${var.bucket_name}-sqs-policy"
  role = "${aws_iam_role.iam_role.id}"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1482274806000",
      "Effect": "Allow",
      "Action": [
        "sqs:ReceiveMessage",
        "sqs:DeleteMessage"
      ],
      "Resource": [
        "arn:aws:sqs:${var.queue_region}:${var.queue_account_number}:${var.queue_name}"
      ]
    },
    {
      "Sid": "AllowExternalS3",
      "Effect": "Allow",
      "Action": [
        "s3:*"
      ],
      "Resource": [
        "arn:aws:s3:::${var.bucket_name}/*"
      ]
    }
  ]
}
EOF
#<<EOF
}

resource "aws_autoscaling_group" "bridge" {
    name = "${var.bucket_name}-bridge-asg"
    availability_zones = ["us-west-2a", "us-west-2b", "us-west-2c"]
    max_size = 20
    wait_for_capacity_timeout = 0
    health_check_grace_period = 300
    health_check_type = "ELB"
    min_size = "${var.node_count}"
    desired_capacity = "${var.node_count}"
    force_delete = true
    vpc_zone_identifier = ["${var.subnet_ids}"]

    launch_configuration = "${aws_launch_configuration.bridge.name}"
    tag {
        key = "Name"
        value = "${var.bucket_name}-bridge"
        propagate_at_launch = true
    }
    tag {
        key = "Owner"
        value = "${var.owner_email}"
        propagate_at_launch = true
    }
    tag {
        key = "Environment"
        value = "${var.environment}"
        propagate_at_launch = true
    }
    tag {
        key = "Service"
        value = "${var.owner}/${var.bucket_name}-bridge"
        propagate_at_launch = true
    }
    load_balancers = ["${aws_elb.bridge.name}"]

    wait_for_elb_capacity = true

    lifecycle {
        create_before_destroy = true
    }
}

resource "aws_launch_configuration" "bridge" {
    name_prefix = "${var.bucket_name}-bridge-launch-configuration-"
    lifecycle {
        create_before_destroy = true
    }
    image_id        = "${var.ami == "default" ? module.baseami.id : var.ami}"
    instance_type   = "${var.instance_type}"
    security_groups = ["${var.security_groups}"]

    associate_public_ip_address = false

    iam_instance_profile = "${aws_iam_instance_profile.instance_profile.id}"

    user_data = "${data.template_cloudinit_config.userdata.rendered}"
}


data "template_file" "cloudinit" {
    template = "${file("${path.module}/cloudinit.yml.tmpl")}"

    vars {
        environment = "${var.environment}"
        service = "${var.bucket_name}-bridge"
    }
}

data "template_cloudinit_config" "userdata" {
    gzip          = false
    base64_encode = false

    part {
        content = "${data.template_file.cloudinit.rendered}"
        content_type = "text/cloud-config"
    }
    part {
        content = "${data.template_file.run_log_bridge.rendered}"
        content_type = "text/x-shellscript"
    }
}

data "template_file" "run_log_bridge" {
    template = "${file("${path.module}/run_log_bridge.sh.tmpl")}"

    vars {
        environment = "${var.environment}"
        service = "${var.bucket_name}-bridge"
        image_version = "${var.image_version}"
        region = "${var.queue_region}"
		queue_account = "${var.queue_account_number}"
		queue_name = "${var.queue_name}"
        es_bulk_kbytes = "${var.es_bulk_kbytes}"
        es_bulk_size = "${var.es_bulk_size}"
        es_bulk_url = "${var.es_bulk_url}"
        es_message_builders = "${var.es_message_builders}"
        es_write_deadline = "${var.es_write_deadline}"
        es_writers = "${var.es_writers}"
        graphite_host = "${var.graphite_host}"
        graphite_interval = "${var.graphite_interval}"
        graphite_prefix = "${var.graphite_prefix}"
        s3_workers = "${var.s3_workers}"
        sqs_workers = "${var.sqs_workers}"
        write_timeout = "${var.write_timeout}"
        extra_options = "${var.extra_options}"

        syslog_host = "${var.syslog_host}"
        syslog_port = "${var.syslog_port}"
        syslog_tag = "${var.syslog_tag}"
        syslog_facility = "${var.syslog_facility}"

        puppet_environment = "${var.environment}"
        puppet_twitch_role = "${var.puppet_twitch_role}"
        puppet_cluster = "${var.puppet_cluster}"
        puppet_team = "${var.puppet_team}"
    }
}

resource "aws_elb" "bridge" {

    # normal
    #  format("%s-bridge", var.bucket_name)
    # long
    #  format("%s-%s-bridge",substr(var.bucket_name,0,16),substr(sha256(var.bucket_name),0,8))

    name = "${length(format("%s-bridge", var.bucket_name)) > 32 ? format("%s-%s-bridge",substr(var.bucket_name,0,min(16, length(var.bucket_name))),substr(sha256(var.bucket_name),0,8)) : format("%s-bridge", var.bucket_name)}"
    subnets = ["${var.subnet_ids}"]
    security_groups = ["${var.security_groups}"]
    internal = true

    listener {
        lb_port = 6060
        lb_protocol = "http"
        instance_port = 6060
        instance_protocol = "http"
    }

    health_check {
        target = "HTTP:6060/health"
        timeout = 5
        interval = 30
        healthy_threshold = 2
        unhealthy_threshold = 2
    }

    cross_zone_load_balancing = true
    idle_timeout = 300
}
