data "aws_caller_identity" "current" { }

output "account_id" {
  value = "${data.aws_caller_identity.current.account_id}"
}

variable "service" {
}

variable "environment" {
}

variable "queue_region" {
}

variable "queue_account_number" {
}

variable "queue_name" {
}

variable "bucket_name" {
}

variable "es_bulk_url" {}

variable "lambda_memory_size" {
    default = "256"
}

data "aws_s3_bucket_object" "lambda" {
  bucket = "twitch-central-lambda-source"
  key = "external-s3-bucket-notification-handler.zip"
}


#resource "aws_s3_bucket" "source_code_bucket" {
#	bucket = "${var.service}-${var.environment}-lambda-source"
#	   acl = "private"
#        region = "us-west-2"
#}
#
#resource "aws_s3_bucket_object" "object" {
#        bucket = "${aws_s3_bucket.source_code_bucket.id}"
#           key = "external-s3-bucket-notification-handler.zip"
#        source = "${path.module}/lambda_function_payload.zip"
#          etag = "${md5(file("${path.module}/lambda_function_payload.zip"))}"
#    depends_on = ["null_resource.lambda_zip_builder"]
#}

resource "aws_iam_role" "iam_for_lambda" {
    name = "lambda-${var.queue_account_number}-${var.queue_name}-s3"
    assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Effect": "Allow"
    }
  ]
}
EOF
}

#resource "null_resource" "lambda_zip_builder" {
#	triggers {
#		lambda_function_hash = "${base64sha256(file("${path.module}/lambda_function.py"))}"
#	}
#
#	provisioner "local-exec" {
#		command = "cd ${path.module}; rm -f lambda_function_payload.zip; zip lambda_function_payload.zip lambda_function.py"
#	}
#}

data "aws_security_group" "sg" {
    name = "twitch_subnets"
}

data "aws_subnet" "subnet_a" { tags { "Name" = "Private - A" } }
data "aws_subnet" "subnet_b" { tags { "Name" = "Private - B" } }
data "aws_subnet" "subnet_c" { tags { "Name" = "Private - C" } }

resource "aws_lambda_function" "test_lambda" {
    function_name = "${var.queue_region}-${var.queue_account_number}-${var.queue_name}"
    vpc_config = {
      subnet_ids = [
        "${data.aws_subnet.subnet_a.id}",
        "${data.aws_subnet.subnet_b.id}",
        "${data.aws_subnet.subnet_c.id}",
      ]
      security_group_ids = ["${data.aws_security_group.sg.id}"]
    }
    memory_size = "${var.lambda_memory_size}"
    role = "${aws_iam_role.iam_for_lambda.arn}"
    handler = "lambda_function.lambda_handler"
    timeout = 240
    s3_bucket = "${data.aws_s3_bucket_object.lambda.bucket}"
    s3_key = "${data.aws_s3_bucket_object.lambda.key}"
    s3_object_version = "${data.aws_s3_bucket_object.lambda.version_id}"
    #source_code_hash = "${base64sha256(file("${path.module}/lambda_function_payload.zip"))}"
    runtime = "python2.7"
    environment {
        variables = {
	    QUEUE_URL = "https://sqs.${var.queue_region}.amazonaws.com/${var.queue_account_number}/${var.queue_name}"
	    FUNCTION_NAME = "${var.queue_region}-${var.queue_account_number}-${var.queue_name}"
            ES_BULK_URL = "${var.es_bulk_url}"
        }
    }
}

resource "aws_iam_role_policy" "sqs_policy" {
  name = "${var.service}-${var.environment}-${var.queue_region}-${var.queue_account_number}-${var.queue_name}-sqs-policy"
  role = "${aws_iam_role.iam_for_lambda.id}"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1482370582000",
      "Effect": "Allow",
      "Action": [
        "lambda:InvokeAsync",
        "lambda:InvokeFunction"
      ],
      "Resource": [
        "arn:aws:lambda:us-west-2:${data.aws_caller_identity.current.account_id}:function:${var.queue_region}-${var.queue_account_number}-${var.queue_name}"
      ]
    },
    {
      "Sid": "Stmt1482274806000",
      "Effect": "Allow",
      "Action": [
        "sqs:ReceiveMessage",
        "sqs:DeleteMessage"
      ],
      "Resource": [
        "arn:aws:sqs:${var.queue_region}:${var.queue_account_number}:${var.queue_name}"
      ]
    },
    {
      "Sid": "AllowExternalS3",
      "Effect": "Allow",
      "Action": [
        "s3:*"
      ],
      "Resource": [
        "arn:aws:s3:::${var.bucket_name}/*"
      ]
    },
    {
      "Effect": "Allow",
      "Action": [
        "logs:CreateLogGroup",
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ],
      "Resource": "arn:aws:logs:*:*:*"
    },
    {
      "Effect": "Allow",
      "Resource": "*",
      "Action": [
        "ec2:DescribeInstances",
        "ec2:CreateNetworkInterface",
        "ec2:AttachNetworkInterface",
        "ec2:DeleteNetworkInterface",
        "ec2:DescribeNetworkInterfaces",
        "autoscaling:CompleteLifecycleAction"
      ]
    }
  ]
}
EOF
#<<EOF
}


resource "aws_lambda_permission" "allow_cloudwatch_to_call_lambda" {
    statement_id = "AllowExecutionFromCloudWatch"
    action = "lambda:InvokeFunction"
    function_name = "${aws_lambda_function.test_lambda.function_name}"
    principal = "events.amazonaws.com"
    source_arn = "${aws_cloudwatch_event_rule.every_minute.arn}"
}

resource "aws_cloudwatch_event_rule" "every_minute" {
    name = "${aws_lambda_function.test_lambda.function_name}"
    description = "Fires every minute"
    schedule_expression = "rate(1 minute)"
}

resource "aws_cloudwatch_event_target" "every_minute" {
    rule = "${aws_cloudwatch_event_rule.every_minute.name}"
    target_id = "${aws_lambda_function.test_lambda.function_name}"
    arn = "${aws_lambda_function.test_lambda.arn}"
}

