resource "aws_launch_configuration" "data" {
    name_prefix = "${var.service}-launch-configuration-"
    lifecycle {
        create_before_destroy = true
    }
    image_id        = "${var.ami}"
    instance_type   = "i2.2xlarge"
    security_groups = ["${var.security_groups}"]

    associate_public_ip_address = false

    iam_instance_profile = "${aws_iam_instance_profile.instance_profile.id}"

    user_data = "${data.template_cloudinit_config.es_data_userdata.rendered}"
}

data "template_file" "es_data_installer" {
    template = "${file("${path.module}/install_elasticsearch.sh.tmpl")}"

    vars {
        cluster_name = "${var.service}-${var.environment}"
        environment = "${var.environment}"
        puppet_twitch_role = "${var.service}"
        puppet_cluster = "${var.service}"
        puppet_environment = "${lookup(map("prod", "production", "dev", "development"), var.environment, "development")}"
        master_fqdn = "${var.service}-masters.${var.environment}.us-west2.justin.tv"
        graphite_host = "graphite.internal.justin.tv"
        zones = "us-west-2a,us-west-2b,us-west-2c"
        master_node_count = "3"
        master_node = "false"
        data_node = "true"
        ingest_node = "true"
    }
}

data "template_cloudinit_config" "es_data_userdata" {
    gzip          = false
    base64_encode = false

    part {
        content = "${file("${path.module}/setupdisks.sh")}"
        content_type = "text/x-shellscript"
    }
    part {
        content = "${data.template_file.es_cloudinit.rendered}"
        content_type = "text/cloud-config"
    }
    part {
        content = "${data.template_file.es_data_installer.rendered}"
        content_type = "text/x-shellscript"
    }
}

resource "aws_autoscaling_group" "data" {
    name = "${var.service}-data-asg"
    availability_zones = ["us-west-2a", "us-west-2b", "us-west-2c"]
    max_size = 200
    min_size = 1
    wait_for_capacity_timeout = 0
    health_check_grace_period = 1200
    health_check_type = "ELB"
    desired_capacity = "${var.data_nodes}"
    force_delete = true
    vpc_zone_identifier = ["${var.subnet_ids}"]
    protect_from_scale_in = true

    launch_configuration = "${aws_launch_configuration.data.name}"
    tag {
        key = "Name"
        value = "${var.service}-data"
        propagate_at_launch = true
    }
    tag {
        key = "Owner"
        value = "${var.owner_email}"
        propagate_at_launch = true
    }
    tag {
        key = "Environment"
        value = "${var.environment}"
        propagate_at_launch = true
    }
    tag {
        key = "Service"
        value = "${var.owner}/${var.service}-data"
        propagate_at_launch = true
    }
    load_balancers = ["${aws_elb.data.name}"]

    #wait_for_elb_capacity = true

    lifecycle {
        create_before_destroy = true
    }
}

resource "aws_elb" "data" {
    name = "${var.service}-${var.environment}"
    subnets = ["${var.subnet_ids}"]
    security_groups = ["${var.security_groups}"]
    internal = true

    listener {
        lb_port = 9200
        lb_protocol = "http"
        instance_port = 9200
        instance_protocol = "http"
    }

    health_check {
        target = "TCP:9200"
        timeout = 5
        interval = 30
        healthy_threshold = 2
        unhealthy_threshold = 2
    }

    cross_zone_load_balancing = true
    idle_timeout = 300
}

resource "aws_route53_record" "ingest_dns" {
    provider = "aws.twitch-aws"
    zone_id  = "${data.aws_route53_zone.justintv.zone_id}"
    name     = "${var.service}.${var.environment}.us-west2.justin.tv"
    type     = "CNAME"
    ttl      = "60"
    records  = ["${aws_elb.data.dns_name}"]
}

output "es_bulk_url" {
    value = "http://${aws_route53_record.ingest_dns.name}:9200/_bulk"
}
