
data "template_file" "kibana_installer" {
	template = "${file("${path.module}/install_kibana.sh.tmpl")}"

	vars {
		es_elb = "${var.service}.${var.environment}.us-west2.justin.tv"
		puppet_twitch_role = "${var.service}-kibana"
		puppet_cluster = "${var.service}-kibana"
        puppet_environment = "${lookup(map("prod", "production", "dev", "development"), var.environment, "development")}"
	}
}

module "baseami" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/dta/tf_baseami_current?ref=1.0.0"
  datacenter = "us-west-2"
  ami_name = "baseami-xenial"
}

data "template_cloudinit_config" "kibana_userdata" {
  gzip          = false
  base64_encode = false

  part {
    content = "${file("${path.module}/setupdisks.sh")}"
    content_type = "text/x-shellscript"
  }
  part {
    content = "${data.template_file.es_cloudinit.rendered}"
    content_type = "text/cloud-config"
  }
  part {
    content = "${data.template_file.kibana_installer.rendered}"
    content_type = "text/x-shellscript"
  }
}

resource "aws_launch_configuration" "kibana" {
  name_prefix = "${var.service}-kibana-launch-configuration-"
  lifecycle {
    create_before_destroy = true
  }
  image_id        = "${module.baseami.id}"
  instance_type   = "t2.micro"
  security_groups = ["${var.security_groups}"]
  
  associate_public_ip_address = false
  
  iam_instance_profile = "${aws_iam_instance_profile.instance_profile.id}"
  
  user_data = "${data.template_cloudinit_config.kibana_userdata.rendered}"
}

resource "aws_autoscaling_group" "kibana" {
  name = "${var.service}-kibana-asg"
  availability_zones = ["us-west-2a", "us-west-2b", "us-west-2c"]
  wait_for_capacity_timeout = 0
  max_size = 21
  min_size = 1
  health_check_grace_period = 1200
  health_check_type = "ELB"
  desired_capacity = 2
  force_delete = true
  vpc_zone_identifier = ["${var.subnet_ids}"]
  protect_from_scale_in = true
  
  launch_configuration = "${aws_launch_configuration.kibana.name}"
  tag {
    key = "Service"
    value = "${var.owner}/${var.service}-kibana"
    propagate_at_launch = true
  }
  tag {
    key = "Name"
    value = "${var.service}-kibana"
    propagate_at_launch = true
  }
  tag {
    key = "Owner"
    value = "${var.owner_email}"
    propagate_at_launch = true
  }
  tag {
    key = "Environment"
    value = "${var.environment}"
    propagate_at_launch = true
  }
  
  load_balancers = ["${element(compact(concat(aws_elb.kibana.*.name,aws_elb.kibana_ssl.*.name)), 0)}"]
  
  #wait_for_elb_capacity = true
  
  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_elb" "kibana" {
  # If we have an ELB cert ARN, don't make a plain HTTP
  count = "${var.elb_cert_arn == "unset" ? 1 : 0}"
  name = "${var.service}-${var.environment}-kibana"
  subnets = ["${var.subnet_ids}"]
  security_groups = ["${var.security_groups}"]
  internal = true

  listener {
    lb_port = 80
    lb_protocol = "http"
    instance_port = 80
    instance_protocol = "http"
  }
  
  health_check {
    target = "HTTP:80/health"
    timeout = 5
    interval = 30
    healthy_threshold = 2
    unhealthy_threshold = 10
  }
  
  cross_zone_load_balancing = true
  idle_timeout = 300
}

resource "aws_elb" "kibana_ssl" {
  # Make this if we have a cert ARN
  count = "${var.elb_cert_arn == "unset" ? 0 : 1}"
  name = "${var.service}-${var.environment}-kibana"
  subnets = ["${var.subnet_ids}"]
  security_groups = ["${var.security_groups}"]
  internal = true
  
  listener {
    lb_port = 80
    lb_protocol = "http"
    instance_port = 80
    instance_protocol = "http"
  }

  # If we have an elb_cert_arn, make an SSL endpoint
  listener {
    lb_port = 443
    lb_protocol = "https"
    instance_port = 80
    instance_protocol = "http"
    ssl_certificate_id = "${var.elb_cert_arn}"
  }

  health_check {
    target = "HTTP:80/health"
    timeout = 5
    interval = 30
    healthy_threshold = 2
    unhealthy_threshold = 10
  }
  cross_zone_load_balancing = true
  idle_timeout = 300
}

# Non-ssl Route 53 record, because Terraform complains if we reference something non-existant
resource "aws_route53_record" "kibana" {
  count = "${var.elb_cert_arn == "unset" ? 1 : 0}"
  provider = "aws.twitch-aws"
  zone_id  = "${data.aws_route53_zone.justintv.zone_id}"
  name     = "${var.service}-kibana.${var.environment}.us-west2.justin.tv"
  type     = "CNAME"
  ttl      = "60"
  # Make the Route53 record match the ELB
  records = ["${aws_elb.kibana.dns_name}"]
}

# SSL Route53 record
resource "aws_route53_record" "kibana_ssl" {
  count = "${var.elb_cert_arn == "unset" ? 0 : 1 }"
  provider = "aws.twitch-aws"
  zone_id  = "${data.aws_route53_zone.justintv.zone_id}"
  name     = "${var.service}-kibana.${var.environment}.us-west2.justin.tv"
  type     = "CNAME"
  ttl      = "60"
  # Make the Route53 record match the ELB
  records = ["${aws_elb.kibana_ssl.dns_name}"]
}
