variable "environment" {}
variable "service" {}
variable "subnet_ids" {type = "list"}
variable "security_groups" {type = "list"}
variable "owner" {}
variable "owner_email" {}
variable "azs" {type = "list"}
variable "deploy_template" {
    default = "yes"
    description = "set this to no to prevent ES templates from being deployed (overwritten) on the ES cluster."
}
variable "ami" {
    default = "ami-c962d1b1"
}
variable "master_a_ami" {
    default = "ami-c962d1b1"
}
variable "master_b_ami" {
    default = "ami-c962d1b1"
}
variable "master_c_ami" {
    default = "ami-c962d1b1"
}
variable "data_nodes" {
    default = 3
}
variable "diamond_image_version" {
    default = "latest"
}
variable "elb_cert_arn" {
  default = "unset"
}

provider "aws" {
  alias   = "twitch-aws"
  profile = "twitch-aws"
  region  = "us-west-2"
}

resource "aws_iam_role" "iam_role" {
  name = "${var.service}.${var.environment}"
  path = "/"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
#<<EOF
}

data "template_file" "es_cloudinit" {
  template = "${file("${path.module}/cloudinit.yml.tmpl")}"

  vars {
    environment = "${var.environment}"
    service = "${var.service}"
  }
}

data "template_file" "es_master_installer" {
  template = "${file("${path.module}/install_elasticsearch.sh.tmpl")}"

  vars {
    cluster_name = "${var.service}-${var.environment}"
    environment = "${var.environment}"
    puppet_twitch_role = "${var.service}"
    puppet_cluster = "${var.service}"
    puppet_environment = "${lookup(map("prod", "production", "dev", "development"), var.environment, "development")}"
    master_fqdn = "${var.service}-masters.${var.environment}.us-west2.justin.tv"
    graphite_host = "graphite.internal.justin.tv"
    zones = "${join(",",var.azs)}"
    master_node_count = "3"
    master_node = "true"
    data_node = "false"
    ingest_node = "false"
  }
}

data "template_cloudinit_config" "es_master_userdata" {
  gzip          = false
  base64_encode = false

  part {
    content = "${file("${path.module}/setupdisks.sh")}"
    content_type = "text/x-shellscript"
  }
  part {
    content = "${data.template_file.es_cloudinit.rendered}"
    content_type = "text/cloud-config"
  }
  part {
    content = "${data.template_file.es_master_installer.rendered}"
    content_type = "text/x-shellscript"
  }
}

resource "aws_instance" "master_a" {
  ami = "${var.master_a_ami}"
  instance_type = "r4.2xlarge"
  subnet_id = "${element(var.subnet_ids, 0)}"
  lifecycle {
    ignore_changes = ["user_data", "ami"]
  }
  associate_public_ip_address = false
  iam_instance_profile = "${aws_iam_instance_profile.instance_profile.id}"
  vpc_security_group_ids = ["${var.security_groups}"]
  user_data = "${data.template_cloudinit_config.es_master_userdata.rendered}"
  tags {
    Name = "${var.service}"
    Owner = "${var.owner_email}"
    Environment = "${var.environment}"
    Service = "${var.owner}/${var.service}-master"
  }
}

resource "aws_instance" "master_b" {
  ami = "${var.master_b_ami}"
  instance_type = "r4.2xlarge"
  subnet_id = "${element(var.subnet_ids, 1)}"
  lifecycle {
    ignore_changes = ["user_data", "ami"]
  }
  associate_public_ip_address = false
  iam_instance_profile = "${aws_iam_instance_profile.instance_profile.id}"
  vpc_security_group_ids = ["${var.security_groups}"]
  user_data = "${data.template_cloudinit_config.es_master_userdata.rendered}"
  tags {
    Name = "${var.service}"
    Owner = "${var.owner_email}"
    Environment = "${var.environment}"
    Service = "${var.owner}/${var.service}-master"
  }
}

resource "aws_instance" "master_c" {
  ami = "${var.master_c_ami}"
  instance_type = "r4.2xlarge"
  subnet_id = "${element(var.subnet_ids, 2)}"
  lifecycle {
    ignore_changes = ["user_data", "ami"]
  }
  associate_public_ip_address = false
  iam_instance_profile = "${aws_iam_instance_profile.instance_profile.id}"
  vpc_security_group_ids = ["${var.security_groups}"]
  user_data = "${data.template_cloudinit_config.es_master_userdata.rendered}"
  tags {
    Name = "${var.service}"
    Owner = "${var.owner_email}"
    Environment = "${var.environment}"
    Service = "${var.owner}/${var.service}-master"
  }
}

resource "null_resource" "deploy_es_template_a" {
    triggers {
        master_a = "${var.deploy_template}${aws_instance.master_a.id}"
    }

    provisioner "local-exec" {
        command = <<EOF
if [ x"${var.deploy_template}" != xyes ];then
     exit 0
fi
until curl -s 'http://${aws_instance.master_a.private_ip}:9200/_cat/health' | awk '{print $4}'|grep -q green; do sleep 1;done
curl -XPUT -d '@${path.module}/template.json' 'http://${aws_instance.master_a.private_ip}:9200/_template/template_a'
EOF
    }
}
resource "null_resource" "deploy_es_template_b" {
    triggers {
        master_b = "${var.deploy_template}${aws_instance.master_b.id}"
    }

    provisioner "local-exec" {
        command = <<EOF
if [ x"${var.deploy_template}" != xyes ];then
     exit 0
fi
until curl -s 'http://${aws_instance.master_b.private_ip}:9200/_cat/health' | awk '{print $4}'|grep -q green; do sleep 1;done
curl -XPUT -d '@${path.module}/template.json' 'http://${aws_instance.master_b.private_ip}:9200/_template/template_a'
EOF
    }
}
resource "null_resource" "deploy_es_template_c" {
    triggers {
        master_c = "${var.deploy_template}${aws_instance.master_c.id}"
    }

    provisioner "local-exec" {
        command = <<EOF
if [ x"${var.deploy_template}" != xyes ];then
     exit 0
fi
until curl -s 'http://${aws_instance.master_c.private_ip}:9200/_cat/health' | awk '{print $4}'|grep -q green; do sleep 1;done
curl -XPUT -d '@${path.module}/template.json' 'http://${aws_instance.master_c.private_ip}:9200/_template/template_a'
EOF
    }
}

data "aws_route53_zone" "justintv" {
  provider = "aws.twitch-aws"
  name = "us-west2.justin.tv."
  private_zone = true
}

resource "aws_route53_record" "master_dns" {
  provider = "aws.twitch-aws"
  zone_id  = "${data.aws_route53_zone.justintv.zone_id}"
  name     = "${var.service}-masters.${var.environment}.us-west2.justin.tv"
  type     = "A"
  ttl      = "60"
  records  = ["${aws_instance.master_a.private_ip}","${aws_instance.master_b.private_ip}","${aws_instance.master_c.private_ip}"]
}

resource "aws_iam_instance_profile" "instance_profile" {
  name = "${var.service}-${var.environment}"
  roles = ["${aws_iam_role.iam_role.id}"]
  depends_on = ["aws_iam_role.iam_role"]
}
#module "pagerduty" {
#  source = "vendor/modules/http-elb-healthchecks"
#  account = "twitch-central-dev"
#  elb_name = "${var.service}-${var.environment}"
#  healthy_backend_threshold = 1
#  rate_5xx_threshold = 500
#  latency_threshold = 3000
#  pagerduty_cloudwatch_endpoint = "https://events.pagerduty.com/integration/b5a21b560d1846898494ea5dc923c2e8/enqueue"
#}
# register fqdn as an output variable
#output "fqdn" {
#  value = "${aws_route53_record.dns.name}"
#}

data "template_file" "logstash_installer" {
  template = "${file("${path.module}/install_logstash.sh.tmpl")}"
  vars {
    puppet_twitch_role = "${var.service}"
    puppet_cluster = "${var.service}"
    puppet_environment = "${lookup(map("prod", "production", "dev", "development"), var.environment, "development")}"
  }
}
