provider "aws" {
  alias   = "${var.account}"
  region  = "${var.region}"
  profile = "${var.account}"
}

# ELB w/ 2 listeners.
resource "aws_elb" "elb" {
  provider = "aws.${var.account}"
  name     = "${var.service}-${var.environment}"
  internal = true

  # There is unfortunately no other way to create two listeners while one is optional.
  # What we do here is create a dummy listener on port 65535 if port80 is not true.
  listener = [
    {
      instance_port      = "${local.instance_port}"
      instance_protocol  = "${local.instance_protocol}"
      lb_port            = "${var.lb_port}"
      lb_protocol        = "${var.lb_protocol}"
      ssl_certificate_id = "${var.ssl_arn}"
    },
    {
      instance_port     = "${var.port80 == "true" ? local.instance_port : "65535"}"
      instance_protocol = "${var.port80 == "true" ? local.instance_protocol : "TCP"}"
      lb_port           = "${var.port80 == "true" ? "80" : "65535"}"
      lb_protocol       = "${var.port80 == "true" ? "HTTP" : "TCP"}"
    },
  ]

  tags {
    Name        = "${var.service}-${var.environment}-elb"
    Service     = "${var.service}"
    Environment = "${var.environment}"
    Src         = "terraform-modules/elb.git"
  }

  subnets         = ["${var.subnets}"]
  security_groups = ["${var.security_groups}"]
  instances       = ["${var.instances}"]
}

# DNS FOR ELB - Create record with, or record without the region name.
provider "aws" {
  alias   = "elb-dns-provider"
  region  = "us-west-2"
  profile = "${var.r53_profile == "" ? var.account : var.r53_profile}"
}

locals {
  # Dynamically generate the dns record: service.env.region, service.env, service.region, service
  env_tags          = "${map("production","prod",  "staging","stage",  "development","dev")}"
  env_dns           = "${lookup(local.env_tags, var.environment, var.environment)}"
  dns_recenv        = "${var.service}${var.env_dns == "false" ? "" : ".${local.env_dns}" }"
  dns_record        = "${local.dns_recenv}${var.regional_dns == "false" ? "" : ".${var.region}" }"
  instance_port     = "${var.instance_port == "" ? var.lb_port : var.instance_port}"
  instance_protocol = "${var.instance_protocol == "" ? var.lb_protocol : var.instance_protocol}"
}

resource "aws_route53_record" "elb_dns" {
  provider = "aws.elb-dns-provider"
  zone_id  = "${var.r53_zone_id}"
  name     = "${local.dns_record}"
  type     = "A"

  alias {
    name                   = "${aws_elb.elb.dns_name}"
    zone_id                = "${aws_elb.elb.zone_id}"
    evaluate_target_health = "${var.evaluate_target_health}"
  }
}
