# Setup AWS account
provider "aws" {
  alias = "${var.account}"
  region = "${var.region}"
  profile = "${var.account}"
}

# SNS Topic for the ELB
resource "aws_sns_topic" "elb_alert_sns" {
  provider = "aws.${var.account}"
  name = "${var.elb_name}"
}

# Subscribe PagerDuty to the SNS Topic
resource "aws_sns_topic_subscription" "pagerduty_sns_subscription" {
  provider = "aws.${var.account}"
  topic_arn = "${aws_sns_topic.elb_alert_sns.arn}"
  protocol = "https"
  endpoint = "${var.pagerduty_cloudwatch_endpoint}"
  endpoint_auto_confirms = true
}

# Create an alarm for when there are no healthy backends
resource "aws_cloudwatch_metric_alarm" "elb_healthy_backends_alarm" {
  provider = "aws.${var.account}"
  alarm_name = "${var.elb_name}-healthy-backends"
  comparison_operator = "LessThanThreshold"
  evaluation_periods = "${var.evaluation_periods}"
  metric_name = "HealthyHostCount"
  namespace = "AWS/ELB"
  period = "${var.check_period}"
  statistic = "Average"
  threshold = "${var.healthy_backend_threshold}"
  alarm_description = "Triggered when there are fewer than ${var.healthy_backend_threshold} healthy backend instances attached to this elb"
  alarm_actions = ["${aws_sns_topic.elb_alert_sns.arn}"]
  ok_actions = ["${aws_sns_topic.elb_alert_sns.arn}"]
  dimensions {
    LoadBalancerName = "${var.elb_name}"
  }
}

# Create an alarm for when there is a spike in 5XX's
resource "aws_cloudwatch_metric_alarm" "elb_rate_5xx_alarm" {
  provider = "aws.${var.account}"
  alarm_name = "${var.elb_name}-5xx-rate"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods = "${var.evaluation_periods}"
  metric_name = "HTTPCode_Backend_5XX"
  namespace = "AWS/ELB"
  period = "${var.check_period}"
  statistic = "Sum"
  threshold = "${var.rate_5xx_threshold}"
  alarm_description = "Triggered when there are more than ${var.rate_5xx_threshold} 5XX's per minute"
  alarm_actions = ["${aws_sns_topic.elb_alert_sns.arn}"]
  ok_actions = ["${aws_sns_topic.elb_alert_sns.arn}"]
  dimensions {
    LoadBalancerName = "${var.elb_name}"
  }
}

# Create an alarm for when latency gets too high
resource "aws_cloudwatch_metric_alarm" "elb_latency_alarm" {
  provider = "aws.${var.account}"
  alarm_name = "${var.elb_name}-latency"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods = "${var.evaluation_periods}"
  metric_name = "Latency"
  namespace = "AWS/ELB"
  period = "${var.check_period}"
  statistic = "Average"
  threshold = "${var.latency_threshold}"  # Milliseconds
  alarm_description = "Triggered when latency rises above ${var.latency_threshold} milliseconds on average"
  alarm_actions = ["${aws_sns_topic.elb_alert_sns.arn}"]
  ok_actions = ["${aws_sns_topic.elb_alert_sns.arn}"]
  dimensions {
    LoadBalancerName = "${var.elb_name}"
  }
}
