provider "aws" {
  alias   = "twitch-account-name"
  profile = var.account
  region  = var.region
}

# Via https://git.xarth.tv/achou/privatelink/blob/master/terraform/modules/privatelink.tf
resource "aws_lambda_function" "pl_lambda_function" {
  provider      = aws.twitch-account-name
  function_name = "${var.service}-${var.environment}-PrivateLinkLambda"
  s3_bucket     = "exampleloadbalancer-us-west-2"
  s3_key        = "blog-posts/static-ip-for-application-load-balancer/populate_NLB_TG_with_ALB.zip"
  description   = "Register Application Load Balancer to Network Load Balancer"
  handler       = "populate_NLB_TG_with_ALB.lambda_handler"
  role          = aws_iam_role.pl_lambda_iam_role.arn
  runtime       = "python2.7"
  timeout       = "300"

  environment {
    variables = {
      ALB_DNS_NAME                      = var.dns_name
      ALB_LISTENER                      = var.lb_port
      CW_METRIC_FLAG_IP_COUNT           = "True"
      INVOCATIONS_BEFORE_DEREGISTRATION = "3"
      MAX_LOOKUP_PER_INVOCATION         = "50"
      NLB_TG_ARN                        = aws_lb_target_group.pl_target_group.arn
      S3_BUCKET                         = local.s3_bucket
    }
  }

  tags = {
    Src = "terraform-modules/privatelink-alb.git"
    Env = var.environment
    Svc = "${var.service} ${var.dns_name}:${var.lb_port}"
  }
}

resource "aws_iam_role" "pl_lambda_iam_role" {
  provider = aws.twitch-account-name
  name     = "${var.service}-${var.environment}-PrivateLink"

  tags = {
    Src = "terraform-modules/privatelink-alb.git"
    Env = var.environment
    Svc = "${var.service} ${var.dns_name}:${var.lb_port}"
  }

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": {
    "Effect": "Allow",
    "Principal": {
      "Service": "lambda.amazonaws.com"
    },
    "Action": "sts:AssumeRole"
  }
}
EOF
}

resource "aws_iam_role_policy" "pl_lambda_iam_role_policy" {
  provider = aws.twitch-account-name
  name     = "Lambda-ALBasTarget"
  role     = aws_iam_role.pl_lambda_iam_role.id

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [{
      "Sid": "LambdaLogging",
      "Effect": "Allow",
      "Action": [
        "logs:CreateLogGroup",
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ],
      "Resource": [
        "arn:aws:logs:*:*:*"
      ]
    },
    {
      "Sid": "S3",
      "Action": [
        "s3:Get*",
        "s3:PutObject",
        "s3:CreateBucket",
        "s3:ListBucket",
        "s3:ListAllMyBuckets"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "ELB",
      "Action": [
        "elasticloadbalancing:Describe*",
        "elasticloadbalancing:RegisterTargets",
        "elasticloadbalancing:DeregisterTargets"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "CW",
      "Action": [
        "cloudwatch:putMetricData"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_lambda_permission" "pl_lambda_permission" {
  provider      = aws.twitch-account-name
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.pl_lambda_function.function_name
  principal     = "events.amazonaws.com"
  source_arn    = aws_cloudwatch_event_rule.pl_event_rule.arn
}

resource "aws_cloudwatch_event_rule" "pl_event_rule" {
  provider            = aws.twitch-account-name
  description         = "ScheduledRule"
  schedule_expression = "rate(1 minute)"

  tags = {
    Src = "terraform-modules/privatelink-alb.git"
    Env = var.environment
    Svc = "${var.service} ${var.dns_name}:${var.lb_port}"
  }
}

resource "aws_cloudwatch_event_target" "pl_event_target" {
  provider = aws.twitch-account-name
  rule     = aws_cloudwatch_event_rule.pl_event_rule.name
  arn      = aws_lambda_function.pl_lambda_function.arn
}

resource "aws_lb" "pl_nlb" {
  provider                         = aws.twitch-account-name
  name                             = "${var.service}-${var.environment}-PLNLB"
  internal                         = true
  load_balancer_type               = "network"
  enable_cross_zone_load_balancing = true
  subnets                          = var.subnets

  tags = {
    Src = "terraform-modules/privatelink-alb.git"
    Env = var.environment
    Svc = "${var.service} ${var.dns_name}:${var.lb_port}"
  }
}

resource "aws_lb_target_group" "pl_target_group" {
  provider             = aws.twitch-account-name
  name                 = "${var.service}-${var.environment}-PLTG"
  port                 = var.lb_port
  protocol             = "TCP"
  vpc_id               = var.vpc
  target_type          = "ip"
  deregistration_delay = 300

  health_check {
    healthy_threshold   = 3
    unhealthy_threshold = 3
    protocol            = "TCP"
  }

  tags = {
    Src = "terraform-modules/privatelink-alb.git"
    Env = var.environment
    Svc = "${var.service} ${var.dns_name}:${var.lb_port}"
  }
}

resource "aws_lb_listener" "pl_listener" {
  provider          = aws.twitch-account-name
  load_balancer_arn = aws_lb.pl_nlb.arn
  port              = var.lb_port
  protocol          = "TCP"

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.pl_target_group.arn
  }
}

locals {
  ap = split(",", var.acceptance_required && var.vpc_endpoint_service_allowed_principals[0] == "" ?
    "*" :
    join(",",var.vpc_endpoint_service_allowed_principals) )

  s3_bucket = "${var.s3_bucket == "" ? var.account : var.s3_bucket}"
}

resource "aws_vpc_endpoint_service" "vpc_endpoint_service" {
  provider                   = aws.twitch-account-name
  acceptance_required        = var.acceptance_required
  network_load_balancer_arns = [aws_lb.pl_nlb.arn]

  allowed_principals = local.ap
}
