# Use twitch-aws for DNS purposes
provider "aws" {
  alias = "twitch-aws"
  region = "us-west-2"
  profile = "twitch-aws"
}

provider "aws" {
  alias = "${var.account}"
  region = "${var.region}"
  profile = "${var.account}"
}

variable "template_name" {
  default = "user_data.tpl"
}

# Load user_data for each retention level
data "template_file" "user_data" {
  template = "${file(join("/", list(path.module, var.template_name)))}"
  vars {
    team = "${var.team}"
    role = "prometheus"
    env = "${lookup(var.environment_shortname, var.environment)}"
    owner = "${var.owner}"
    environment = "${var.environment}"
    puppet = "${lookup(var.puppet_server, var.puppet)}"
    dns_region = "${lookup(var.dns_region, var.region)}"
  }
}

data "template_cloudinit_config" "userdata" {
  gzip          = false
  base64_encode = false

  part {
    content = "${file("${path.module}/setupdisks.sh")}"
    content_type = "text/x-shellscript"
  }
  part {
    content = "${data.template_file.user_data.rendered}"
    content_type = "text/cloud-config"
  }
}


# AWS Instances

# Retention level 1
resource "aws_instance" "instances" {
  provider = "aws.${var.account}"
  ami = "${var.ami}"
  count = "2"
  instance_type = "${lookup(var.instance_type, var.size)}"
  vpc_security_group_ids = ["${var.vpc_security_group_ids}"]
  # Stripe instances across the specified subnets
  subnet_id = "${element(var.subnet_ids, count.index % length(var.subnet_ids))}"
  root_block_device {
    volume_size = "${var.root_block_device_size}"
  }
  ebs_block_device {
    device_name = "/dev/xvdb"
    volume_type = "gp2"
    volume_size = "${lookup(var.disk_size, var.size)}"
  }
  iam_instance_profile = "${var.instance_profile}"
  # Render and pass in the user_data template
  user_data = "${data.template_cloudinit_config.userdata.rendered}"
  tags {
    Name = "prometheus-${var.environment}"
    Environment = "${lookup(var.environment_tagname, var.environment)}"
    Owner = "${var.owner}@twitch.tv"
    Service = "${var.owner}/prometheus/prometheus"
  }
  # Set a tag for use with codedeploy.  The Repo used for deployment should == role of these instances
  provisioner "local-exec" {
    command = "aws ec2 create-tags --resources ${self.id} --tags Key=instance-${var.owner}-prometheus-${var.environment},Value=1 --profile=${var.account} --region=${var.region}"
  }
  lifecycle {
    ignore_changes = ["tags"]
  }
}

# DNS records for each host
resource "aws_route53_record" "instances_dns" {
  provider = "aws.twitch-aws" # We need to host DNS entries in twitch-aws
  count = "2"
  zone_id = "${var.r53_zone_id}"
  name = "prometheus-${replace(element(aws_instance.instances.*.id, count.index), "i-", "")}.${lookup(var.environment_shortname, var.environment)}"
  type = "A"
  ttl = "300"
  records = ["${element(aws_instance.instances.*.private_ip, count.index)}"]
}

# Load Balancer
resource "aws_lb" "load_balancer" {
  provider        = "aws.${var.account}"
  name            = "prometheus-${var.team}"
  internal        = true
	load_balancer_type = "application"
  security_groups = ["${var.vpc_security_group_ids}"]
  subnets         = ["${var.subnet_ids}"]
  ip_address_type = "ipv4"

  enable_deletion_protection = true

  tags {
		Name = "prometheus-${var.team}"
    Environment = "production"
    Owner = "${var.owner}@twitch.tv"
    Service = "${var.owner}/prometheus/prometheus"
  }
}

# Get the cert info, so we can do HTTPS things
data "aws_acm_certificate" "lb_cert" {
  provider = "aws.${var.account}"
  domain   = "prometheus-${var.team}.internal.justin.tv"
  statuses = ["ISSUED"]
}

resource "aws_lb_listener" "https" {
  provider        = "aws.${var.account}"
  load_balancer_arn = "${aws_lb.load_balancer.arn}"
  port              = "443"
  protocol          = "HTTPS"
  ssl_policy        = "ELBSecurityPolicy-2015-05"
  certificate_arn   = "${data.aws_acm_certificate.lb_cert.arn}"

  default_action {
    target_group_arn = "${aws_lb_target_group.web.arn}"
    type             = "forward"
  }
}

resource "aws_lb_target_group" "web" {
  provider        = "aws.${var.account}"
  port     = "9090"
  protocol = "HTTP"
  vpc_id   = "${lookup(var.vpc_id, var.environment)}"
}

resource "aws_lb_target_group_attachment" "web" {
  provider        = "aws.${var.account}"
  target_group_arn = "${aws_lb_target_group.web.arn}"
  target_id        = "${element(aws_instance.instances.*.id, count.index)}"
	count						 = "2"
}

