### Please see the Readme before modifying this file
locals {
  bucket = "${var.bucket != "" ? var.bucket : "${data.aws_caller_identity.current.account_id}-route53-backup" }"
}

### Define S3 bucket where the backups where go. Lifecycle policy set to 1 yr
resource "aws_s3_bucket" "backup_location" {
  bucket = "${local.bucket}"
  acl    = "private"

  tags = {
    Name = "Route53 Backups"
    Src  = "git/terraform-modules/route53-backups"
  }

  lifecycle_rule {
    id      = "yearly_retention"
    prefix  = "route53_backups/"
    enabled = true

    expiration {
      days = 365
    }

    noncurrent_version_expiration {
      days = 365
    }
  }
}

resource "aws_s3_bucket_object" "backup_folder" {
  bucket = "${aws_s3_bucket.backup_location.id}"
  key    = "route53_backups/"
}

###Lambda function
resource "aws_lambda_function" "lambda" {
  function_name = "route53_backup"

  filename         = "${data.archive_file.zip.output_path}"
  source_code_hash = "${data.archive_file.zip.output_base64sha256}"

  role    = "${aws_iam_role.iam_route53_backup.arn}"
  handler = "route53_lambda_backup.lambda_handler"
  runtime = "python3.7"
  timeout = "300"

  environment {
    variables = {
      bucket_name   = "${aws_s3_bucket.backup_location.id}"
      bucket_region = "${aws_s3_bucket.backup_location.region}"
    }
  }
}

data "archive_file" "zip" {
  type        = "zip"
  source_file = "${path.module}/route53_lambda_backup.py"
  output_path = "route53_lambda_backup.zip"
}

## Cloud watch event defined to trigger lambda every 24 hours

resource "aws_cloudwatch_event_rule" "every_24_hours" {
  name                = "route53_lambda_backup_trigger"
  description         = "Fires every 24 hours"
  schedule_expression = "rate(24 hours)"
}

resource "aws_cloudwatch_event_target" "backup_lambda_every_24_hours" {
  rule = "${aws_cloudwatch_event_rule.every_24_hours.name}"
  arn  = "${aws_lambda_function.lambda.arn}"
}

resource "aws_lambda_permission" "allow_cloudwatch_to_call_check" {
  statement_id  = "AllowExecutionFromCloudWatch"
  action        = "lambda:InvokeFunction"
  function_name = "${aws_lambda_function.lambda.function_name}"
  principal     = "events.amazonaws.com"
  source_arn    = "${aws_cloudwatch_event_rule.every_24_hours.arn}"
}

## Define IAM Policies for the lambda
data "aws_iam_policy_document" "policy" {
  statement {
    sid    = ""
    effect = "Allow"

    principals {
      identifiers = ["lambda.amazonaws.com"]
      type        = "Service"
    }

    actions = ["sts:AssumeRole"]
  }
}

data "aws_iam_policy_document" "actions" {
  statement {
    sid    = ""
    effect = "Allow"

    actions   = ["s3:Write", "s3:Get*", "s3:List*", "s3:Put*"]
    resources = ["${aws_s3_bucket.backup_location.arn}", "${aws_s3_bucket.backup_location.arn}/*"]
  }

  statement {
    sid    = ""
    effect = "Allow"

    actions   = ["route53:List*", "route53:Get*"]
    resources = ["*"]
  }
}

resource "aws_iam_policy" "route53_backup_policy" {
  name   = "route53_backup_policy"
  policy = "${data.aws_iam_policy_document.actions.json}"
}

resource "aws_iam_role" "iam_route53_backup" {
  name               = "iam_route53_backup"
  assume_role_policy = "${data.aws_iam_policy_document.policy.json}"
}

resource "aws_iam_role_policy_attachment" "route53_backup_logging_attachment" {
  role       = "${aws_iam_role.iam_route53_backup.name}"
  policy_arn = "${aws_iam_policy.route53_backup_policy.arn}"
}

## Enable logging to cloudwatch for Lambda

resource "aws_iam_policy" "lambda_logging" {
  name        = "route53_backup_logging"
  path        = "/"
  description = "IAM policy for logging from route53 backup lambda"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "logs:CreateLogGroup",
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ],
      "Resource": "arn:aws:logs:*:*:*",
      "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "lambda_logs" {
  role       = "${aws_iam_role.iam_route53_backup.name}"
  policy_arn = "${aws_iam_policy.lambda_logging.arn}"
}
