provider "aws" {
  alias = "${var.account}"
  region = "${var.region}"
  profile = "${var.account}"
}

resource "aws_kinesis_stream" "stream" {
  provider = "aws.${var.account}"
  name        = "${var.stream_name_prefix}-${var.application_name}"
  shard_count = "${var.shard_count}"

  tags {
    Name  = "${var.stream_name_prefix}-${var.application_name}"
    Owner = "${var.owner}"
  }
}

resource "aws_cloudwatch_metric_alarm" "stream-bytes" {
  provider = "aws.${var.account}"
  alarm_name          = "${var.stream_name_prefix}-${var.application_name}-too-many-bytes"
  alarm_description   = "Too many bytes sent to ${var.stream_name_prefix}-${var.application_name}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingBytes"
  namespace           = "AWS/Kinesis"
  period              = "60"
  statistic           = "Sum"
  threshold           = "${0.85 * var.shard_count * 1000000 * 60}"

  dimensions {
    StreamName = "${aws_kinesis_stream.stream.name}"
  }

  alarm_actions             = ["${var.scieng_sns_topic}", "${aws_sns_topic.stream_alerts.arn}"]
  ok_actions                = ["${var.scieng_sns_topic}", "${aws_sns_topic.stream_alerts.arn}"]
  insufficient_data_actions = ["${var.scieng_sns_topic}", "${aws_sns_topic.stream_alerts.arn}"]
}

resource "aws_cloudwatch_metric_alarm" "stream-records" {
  provider = "aws.${var.account}"
  alarm_name          = "${var.stream_name_prefix}-${var.application_name}-too-many-records"
  alarm_description   = "Too many records sent to ${var.stream_name_prefix}-${var.application_name}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingRecords"
  namespace           = "AWS/Kinesis"
  period              = "60"
  statistic           = "Sum"
  threshold           = "${0.85 * var.shard_count * 1000 * 60}"

  dimensions {
    StreamName = "${aws_kinesis_stream.stream.name}"
  }

  alarm_actions             = ["${var.scieng_sns_topic}", "${aws_sns_topic.stream_alerts.arn}"]
  ok_actions                = ["${var.scieng_sns_topic}", "${aws_sns_topic.stream_alerts.arn}"]
  insufficient_data_actions = ["${var.scieng_sns_topic}", "${aws_sns_topic.stream_alerts.arn}"]
}

# Create monitoring resources for the kinesis streams

# SNS Topic 
resource "aws_sns_topic" "stream_alerts" {
  provider = "aws.${var.account}"
  name = "${var.stream_name_prefix}-${var.application_name}"
}

# Subscribe PagerDuty to the SNS Topic
resource "aws_sns_topic_subscription" "pagerduty_sns_subscription" {
  provider = "aws.${var.account}"
  topic_arn = "${aws_sns_topic.stream_alerts.arn}"
  protocol = "https"
  endpoint = "${var.pagerduty_cloudwatch_endpoint}"
  endpoint_auto_confirms = true
}

resource "aws_iam_role" "spade-putter" {
  provider = "aws.${var.account}"
  name = "${var.stream_name_prefix}-${var.application_name}-putter"
  path = "/"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "AWS": [
          "arn:aws:iam::673385534282:root"
        ]
      },
      "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "spade-putter" {
  provider = "aws.${var.account}"
  name = "${var.stream_name_prefix}-${var.application_name}-putter"
  role = "${aws_iam_role.spade-putter.id}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "kinesis:PutRecords"
      ],
      "Effect": "Allow",
      "Resource": [
        "${aws_kinesis_stream.stream.arn}"
      ]
    }
  ]
}
EOF
}

resource "aws_dynamodb_table" "kinsumer_clients" {
  provider = "aws.${var.account}"
  name           = "${var.stream_name_prefix}-${var.application_name}_clients"
  read_capacity  = "${var.dynamodb_clients_throughput}"
  write_capacity = "${var.dynamodb_clients_throughput}"
  hash_key       = "ID"

  attribute {
    name = "ID"
    type = "S"
  }
}

resource "aws_dynamodb_table" "kinsumer_checkpoints" {
  provider = "aws.${var.account}"
  name           = "${var.stream_name_prefix}-${var.application_name}_checkpoints"
  read_capacity  = "${var.dynamodb_checkpoints_throughput}"
  write_capacity = "${var.dynamodb_checkpoints_throughput}"
  hash_key       = "Shard"

  attribute {
    name = "Shard"
    type = "S"
  }
}

resource "aws_dynamodb_table" "kinsumer_metadata" {
  provider = "aws.${var.account}"
  name           = "${var.stream_name_prefix}-${var.application_name}_metadata"
  read_capacity  = "${var.dynamodb_metadata_throughput}"
  write_capacity = "${var.dynamodb_metadata_throughput}"
  hash_key       = "Key"

  attribute {
    name = "Key"
    type = "S"
  }
}

resource "aws_iam_role_policy" "worker" {
  provider = "aws.${var.account}"
  name = "${var.stream_name_prefix}-${var.application_name}-worker"
  role = "${var.iam_role}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "kinesis:DescribeStream",
        "kinesis:GetShardIterator",
        "kinesis:GetRecords"
      ],
      "Effect": "Allow",
      "Resource": [
        "${aws_kinesis_stream.stream.arn}"
      ]
    },
    {
      "Action": [
        "dynamodb:GetItem",
        "dynamodb:PutItem",
        "dynamodb:DeleteItem",
        "dynamodb:UpdateItem",
        "dynamodb:Scan",
        "dynamodb:DescribeTable"
      ],
      "Effect": "Allow",
      "Resource": [
        "${aws_dynamodb_table.kinsumer_clients.arn}",
        "${aws_dynamodb_table.kinsumer_checkpoints.arn}",
        "${aws_dynamodb_table.kinsumer_metadata.arn}"
      ]
    }
  ]
}
EOF
}


