# force old template provider
provider "template" {
  version = "~> 1.0"
}

data "template_file" "ls_ingress_cloudinit" {
  template = "${file("${path.module}/logstash_ingress.yml.tmpl")}"

  vars {
    environment = "${var.environment}"
    region = "${var.region}"
    patterns_content = "${base64encode(data.template_file.logstash_patterns.rendered)}"
    config_content = "${base64encode(data.template_file.logstash_conf.rendered)}"
    service = "${var.service}"
  }
}

data "template_file" "logstash_installer" {
  template = "${file("${path.module}/install_logstash.sh.tmpl")}"
  
  vars {
    puppet_twitch_role = "${var.service}"
    puppet_cluster = "${var.service}"
    puppet_twitch_team = "${var.twitch_team_fact}"
    puppet_environment = "production"
  }
}

data "template_file" "logstash_conf" {
  template = "${file("${path.module}/logstash.conf.tmpl")}"
  
  vars {
    logstash_inputs_pre = "${var.logstash_inputs_pre}"
    logstash_inputs = "${var.logstash_inputs}"
    logstash_outputs_pre = "${var.logstash_outputs_pre}"
    logstash_outputs = "${var.logstash_outputs}"
    logstash_filters = "${var.logstash_filters}"
    bucket = "${aws_s3_bucket.bucket.id}"
  }
}

data "template_file" "logstash_patterns" {
  template = "${file("${path.module}/patterns.tmpl")}"
  
  vars {
    logstash_patterns = "${var.logstash_patterns}"
  }
}

data "template_cloudinit_config" "ingress_userdata" {
  gzip          = false
  base64_encode = false

  part {
    content = "${file("${path.module}/setupdisks.sh")}"
    content_type = "text/x-shellscript"
  }
  part {
    content = "${data.template_file.ls_ingress_cloudinit.rendered}"
    content_type = "text/cloud-config"
  }
  part {
    content = "${data.template_file.logstash_installer.rendered}"
    content_type = "text/x-shellscript"
  }
}

resource "aws_iam_role" "ingress_iam_role" {
  name = "${var.service}-${var.environment}-ingress"
  path = "/"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
#<<EOF
}

resource "aws_iam_role_policy" "ingress_sqs_policy" { 
  name = "${var.service}-${var.environment}-ingress"
  role = "${aws_iam_role.ingress_iam_role.id}"
  policy = <<EOF
{
  "Statement": [
    {
      "Sid": "Stmt1347986764949",
      "Action": [
        "SQS:Publish"
      ],
      "Effect": "Allow",
      "Resource": [
        "${aws_sqs_queue.queue.arn}"
      ]
   },
   {
      "Sid": "woiejf",
      "Action": [
          "S3:PutObject"
      ],
      "Effect": "Allow",
      "Resource": [
          "arn:aws:s3:::${aws_s3_bucket.bucket.id}/*"
      ]
    }
  ]
}
EOF
#<<EOF
}

resource "aws_launch_configuration" "ingress" {
  name_prefix = "${var.service}-launch-configuration-"
  lifecycle {
    create_before_destroy = true
  }
  image_id        = "${var.ami == "default" ? data.aws_ami.dta.id : var.ami}"
  instance_type   = "c3.4xlarge"
  security_groups = ["${var.security_groups}"]
  
  associate_public_ip_address = false
  
  iam_instance_profile = "${aws_iam_instance_profile.ingress_instance_profile.id}"
  
  user_data = "${data.template_cloudinit_config.ingress_userdata.rendered}"
}

resource "aws_iam_instance_profile" "ingress_instance_profile" {
  name = "${var.service}-${var.environment}-ingress"
  roles = ["${aws_iam_role.ingress_iam_role.id}"]
  depends_on = ["aws_iam_role.ingress_iam_role"]
}

resource "aws_autoscaling_group" "ingress" {
  wait_for_capacity_timeout = 0
  name = "${var.service}-ingress-asg"
  availability_zones = ["${var.azs}"]
  max_size = 21
  min_size = 3
  health_check_grace_period = 1200
  health_check_type = "ELB"
  desired_capacity = "${var.desired_capacity}"
  force_delete = true
  vpc_zone_identifier = ["${var.subnet_ids}"]
  
  launch_configuration = "${aws_launch_configuration.ingress.name}"
  tag {
    key = "Name"
    value = "${var.service}-ingress"
    propagate_at_launch = true
  }
  tag {
	  key = "Service"
	  value = "${var.owner}/${var.service}"
	  propagate_at_launch = true
  }
  tag {
    key = "Owner"
    value = "${var.owner}"
    propagate_at_launch = true
  }
  tag {
    key = "Environment"
    value = "${var.environment}"
    propagate_at_launch = true
  }
  
  load_balancers = ["${aws_elb.ingress.name}"]
  
  #wait_for_elb_capacity = true
  
  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_elb" "ingress" {
  name = "${var.service}-${var.environment}"
  subnets = ["${var.subnet_ids}"]
  security_groups = ["${var.security_groups}"]
  internal = true
  
  listener {
    lb_port = 6514
    lb_protocol = "tcp"
    instance_port = 6514
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 514
    lb_protocol = "tcp"
    instance_port = 6514
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 5044
    lb_protocol = "tcp"
    instance_port = 5044
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7777
    lb_protocol = "tcp"
    instance_port = 7777
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7778
    lb_protocol = "tcp"
    instance_port = 7778
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7779
    lb_protocol = "tcp"
    instance_port = 7779
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7780
    lb_protocol = "tcp"
    instance_port = 7780
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7781
    lb_protocol = "tcp"
    instance_port = 7781
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7782
    lb_protocol = "tcp"
    instance_port = 7782
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7783
    lb_protocol = "tcp"
    instance_port = 7783
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7784
    lb_protocol = "tcp"
    instance_port = 7784
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7785
    lb_protocol = "tcp"
    instance_port = 7785
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7786
    lb_protocol = "tcp"
    instance_port = 7786
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7787
    lb_protocol = "tcp"
    instance_port = 7787
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7788
    lb_protocol = "tcp"
    instance_port = 7788
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7789
    lb_protocol = "tcp"
    instance_port = 7789
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7790
    lb_protocol = "tcp"
    instance_port = 7790
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7791
    lb_protocol = "tcp"
    instance_port = 7791
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7792
    lb_protocol = "tcp"
    instance_port = 7792
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7793
    lb_protocol = "tcp"
    instance_port = 7793
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7794
    lb_protocol = "tcp"
    instance_port = 7794
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7795
    lb_protocol = "tcp"
    instance_port = 7795
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7796
    lb_protocol = "tcp"
    instance_port = 7796
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7797
    lb_protocol = "tcp"
    instance_port = 7797
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7798
    lb_protocol = "tcp"
    instance_port = 7798
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 7799
    lb_protocol = "tcp"
    instance_port = 7799
    instance_protocol = "tcp"
  }
  listener {
    lb_port = 80
    lb_protocol = "http"
    instance_port = 8080
    instance_protocol = "http"
  }
  
  health_check {
    target = "HTTP:8080/healthcheck"
    timeout = 5
    interval = 30
    healthy_threshold = 2
    unhealthy_threshold = 2
  }
  
  cross_zone_load_balancing = true
  idle_timeout = 300
}

data "aws_route53_zone" "justintv" {
  provider = "aws.twitch-aws"
  name = "us-west2.justin.tv."
  private_zone = true
}

resource "aws_route53_record" "ingress" {
  provider = "aws.twitch-aws"
  zone_id  = "${data.aws_route53_zone.justintv.zone_id}"
  name     = "${var.service}.${var.environment}.us-west2.justin.tv"
  type     = "CNAME"
  ttl      = "60"
  records  = ["${aws_elb.ingress.dns_name}"]
}

output "elb_fqdn" {
  value = "${var.service}.${var.environment}.us-west2.justin.tv"
}

