variable "environment" {}

variable "service" {}

variable "central_accounts" {
    type = "map"
    default = {
        prod = "007917851548"
        dev = "386434559102"
    }
}

variable "subnet_ids" {
    type = "list"
}

variable "security_groups" {
    type = "list"
}

variable "owner" {}

variable "owner_email" {}

# What to place for the twitch_team fact.  If this is left unset, nothing will
# be placed at all (not even an empty file).
variable "twitch_team_fact" {
    default = ""
}

variable "region" {
    default = "us-west-2"
}

variable "azs" {
    default = ["us-west-2a", "us-west-2b", "us-west-2c"]
}

variable "desired_capacity" {
    default = "3"
}

variable "visibility_timeout" {
    default = "30"
}

variable "logstash_inputs_pre" {
    default = ""
}

variable "logstash_inputs" {
    default = ""
}

variable "logstash_outputs_pre" {
    default = ""
}

variable "logstash_outputs" {
    default = ""
}

variable "logstash_filters" {
    default = ""
}

variable "logstash_patterns" {
  default = ""
}

data "aws_ami" "dta" {
  provider    = "aws.twitch-aws"
  most_recent = true
  owners      = ["043714768218"]

  filter {
    name   = "name"
    values = ["devtools-baseami-xenial-*"]
  }
}

variable "ami" {
    description = "The AMI to use -- must be able to run apt-get install -y docker.io to install Docker"
    default = "default"
}

provider "aws" {
  alias   = "twitch-aws"
  profile = "twitch-aws"
  region  = "us-west-2"
}

resource "aws_sqs_queue" "queue" {
    name = "${var.service}-${var.environment}-notifications"
    message_retention_seconds = "1209600"
    visibility_timeout_seconds = "${var.visibility_timeout}"
    policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": "*",
      "Action": "sqs:SendMessage",
      "Resource": "arn:aws:sqs:*:*:${var.service}-${var.environment}-notifications",
      "Condition": {
        "ArnEquals": { "aws:SourceArn": "${aws_s3_bucket.bucket.arn}" }
      }
    },
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": "arn:aws:iam::${lookup(var.central_accounts, var.environment)}:root"
      },
      "Action": [
        "SQS:ReceiveMessage",
        "SQS:DeleteMessage",
        "SQS:ChangeMessageVisibility"
      ],
      "Resource": "arn:aws:sqs:*:*:${var.service}-${var.environment}-notifications"
    }
  ]
}
EOF
#<<EOF
}

resource "aws_iam_role" "iam_role" {
  name = "${var.service}-${var.environment}"
  path = "/"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
#<<EOF
}

resource "aws_iam_role_policy" "sqs_policy" { 
  name = "${var.service}-${var.environment}"
  role = "${aws_iam_role.iam_role.id}"
  policy = <<EOF
{
  "Statement": [
    {
      "Sid": "Stmt1347986764948",
      "Action": [
        "sqs:ChangeMessageVisibility",
        "sqs:ChangeMessageVisibilityBatch",
        "sqs:DeleteMessage",
        "sqs:DeleteMessageBatch",
        "sqs:GetQueueAttributes",
        "sqs:GetQueueUrl",
        "sqs:SendMessage",
        "sqs:SendMessageBatch",
        "sqs:ReceiveMessage"
      ],
      "Effect": "Allow",
      "Resource": [
        "${aws_sqs_queue.queue.arn}"
      ]
    }
  ]
}
EOF
#<<EOF
}

resource "aws_iam_instance_profile" "instance_profile" {
  name = "${var.service}-${var.environment}"
  roles = ["${aws_iam_role.iam_role.id}"]
  depends_on = ["aws_iam_role.iam_role"]
}

resource "aws_s3_bucket" "bucket" {
	bucket = "${var.service}-${var.environment}"
	policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Give twitch-central-prod access",
      "Effect": "Allow",
      "Principal": {
        "AWS": "arn:aws:iam::${lookup(var.central_accounts, var.environment)}:root"
      },
      "Action": [
        "s3:GetObject"
      ],
      "Resource": [
        "arn:aws:s3:::${var.service}-${var.environment}/*"
      ]
    }
  ]
}

EOF
#<<EOF
}

resource "aws_s3_bucket_notification" "bucket_notification" {
    bucket = "${aws_s3_bucket.bucket.id}"
    queue {
        queue_arn = "${aws_sqs_queue.queue.arn}"
        events = ["s3:ObjectCreated:*"]
    }
}

output "s3_bucket_name" {
	value = "${aws_s3_bucket.bucket.id}"
}

data "aws_caller_identity" "current" { }

output "queue_account_number" {
  value = "${data.aws_caller_identity.current.account_id}"
}

output "log_bucket" {
  value = "${aws_s3_bucket.bucket.id}"
}

output "sqs_queue_url" {
	value = "${aws_sqs_queue.queue.id}"
}
