package twirptimotyenapp

import (
	"context"
	"math/rand"
	"strings"
	"time"

	"code.justin.tv/hygienic/errors"
	"code.justin.tv/sse/malachai/pkg/s2s/callee"
	"code.justin.tv/timotyenorg/timotyenapp/proto/timotyenapp"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awsutil"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/cep21/circuit"
)

var _ timotyenapp.Timotyenapp = &TimotyenappImpl{}

// TimotyenappImpl implements our example app twirp interface
type TimotyenappImpl struct {
	SecretKey        string
	ItemGetTableName string

	DynamoDB          *dynamodb.DynamoDB
	DownstreamCircuit *circuit.Circuit
	DynamoCircuit     *circuit.Circuit
}

// ItemGet fetches an item from (nothing) and stores current time in dynamodb
func (b *TimotyenappImpl) ItemGet(ctx context.Context, req *timotyenapp.ItemGetRequest) (*timotyenapp.ItemGetResponse, error) {
	authInfo := awsutil.Prettify(callee.GetCallerID(ctx)) + awsutil.Prettify(callee.GetCapabilities(ctx))
	if req.Key == b.SecretKey {
		return &timotyenapp.ItemGetResponse{
			Value:    "YOU GUESSED THE SECRET KEY",
			AuthInfo: authInfo,
		}, nil
	}

	resp := make([]string, 0, 10)
	if err := b.DownstreamCircuit.Run(ctx, func(ctx context.Context) error {
		if strings.Contains(req.Key, "-downstream_failure-") {
			return errors.New("a downstream failure")
		}
		time.Sleep(time.Duration(rand.Float64() * float64(time.Millisecond*100)))
		return nil
	}); err != nil {
		return nil, err
	}

	location, err := time.LoadLocation("America/Los_Angeles")
	if err != nil {
		return nil, err
	}

	setAt := time.Now().In(location)

	putReq, _ := b.DynamoDB.PutItemRequest(&dynamodb.PutItemInput{
		Item: map[string]*dynamodb.AttributeValue{
			"id": {
				S: aws.String(req.GetKey()),
			},
			"setat": {
				S: aws.String(setAt.String()),
			},
		},
		TableName: aws.String(b.ItemGetTableName),
	})
	if err := b.DynamoCircuit.Run(ctx, func(ctx context.Context) error {
		putReq.SetContext(ctx)
		return putReq.Send()
	}); err != nil {
		return nil, err
	}
	return &timotyenapp.ItemGetResponse{
		Value:    strings.Join(resp, ","),
		AuthInfo: authInfo,
	}, nil
}

// HealthCheck verifies the app works
func (b *TimotyenappImpl) HealthCheck() error {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*3)
	defer cancel()
	_, err := b.ItemGet(ctx, &timotyenapp.ItemGetRequest{
		Key: "test_key",
	})
	return err
}
