package main

import (
	"fmt"
	"io"
	"net"
	"net/http"
	"os"
	"strings"
	"time"

	"code.justin.tv/hygienic/twitchbaseservice/v4/basemetrics"
	"code.justin.tv/hygienic/twitchbaseservice/v4/basemetrics/stackcs4"

	"github.com/cep21/circuit"

	"code.justin.tv/hygienic/twirpserviceslohook"
	"code.justin.tv/hygienic/twirpserviceslohook/statsdslo"
	"github.com/twitchtv/twirp"

	"code.justin.tv/hygienic/loglogrus"
	"code.justin.tv/sse/malachai/pkg/events"
	"code.justin.tv/sse/malachai/pkg/s2s/callee"

	"code.justin.tv/hygienic/basehttpserver"
	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/errors"
	"code.justin.tv/hygienic/servicerunner"
	"code.justin.tv/hygienic/twitchbaseservice/v4/stacks5"
	"code.justin.tv/timotyenorg/timotyenapp/cmd/timotyenapp/internal/httpapi"
	"code.justin.tv/timotyenorg/timotyenapp/cmd/timotyenapp/internal/twirptimotyenapp"
	"code.justin.tv/timotyenorg/timotyenapp/proto/timotyenapp"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	twirpstatsd "github.com/twitchtv/twirp/hooks/statsd"
)

var instance = service{
	osExit: os.Exit,
	OsArgs: os.Args[1:],
}

// The name of your service for sandstorm, s2s, etc
const serviceName = "timotyenapp"

// Sandstorm has the concept of a "team".  You want this to be your team inside sandstorm
const serviceTeam = "communications"

type service struct {
	Config         config
	Secrets        secrets
	BaseService    basemetrics.MetricStack
	runner         servicerunner.ServiceRunner
	optionalConfig stacks5.OptionalConfig
	OnListen       func(net.Addr)
	osExit         func(int)
	OsArgs         []string
}

type config struct {
	ItemGetTable    string
	TimotyenappHost string
	S2SAuthService  string
}

func (c *config) Load(d *distconf.Distconf) error {
	c.ItemGetTable = d.Str("item_get_table", "").Get()
	if c.ItemGetTable == "" {
		return errors.New("unable to find item_get_table configuration")
	}
	// You don't have to set this to yourself or anything like that.  This is just an example downstream host.
	// I'm just using myself as that downstream host.
	c.TimotyenappHost = d.Str("timotyenapp_host", "").Get()
	if c.TimotyenappHost == "" {
		return errors.New("unable to find timotyenapp_host configuration")
	}
	c.S2SAuthService = d.Str("s2s_auth", "").Get()
	return nil
}

type secrets struct {
	AppSecret string
}

func (s *secrets) Load(d *distconf.Distconf) error {
	// secrets fetch from sandstorm if you have SANDSTORM_ARN set. If not, will attempt to pull from env
	s.AppSecret = d.Str("timotyenappsecret", "").Get()
	return nil
}

func (s *service) setupAuth() (*callee.Client, error) {
	// Assume config variable "s2s_auth" is your s2s service name.  If it's empty, skips s2s
	if s.Config.S2SAuthService == "" {
		s.BaseService.Logger().Log("skipping s2s auth")
		return nil, nil
	}
	s.BaseService.Logger().Log("auth", s.Config.S2SAuthService, "setting up s2s auth")
	logrusLogger := loglogrus.Log{
		Logger: s.BaseService.Logger(),
		Level:  loglogrus.WarnLevel,
	}
	eventsWriterClient, err := events.NewEventLogger(events.Config{
		BufferSizeLimit: 0,
	}, &logrusLogger)
	if err != nil {
		return nil, errors.Wrap(err, "unable to make s2s event logger")
	}
	ret := &callee.Client{
		ServiceName:        s.Config.S2SAuthService,
		EventsWriterClient: eventsWriterClient,
		Logger:             &logrusLogger,
		// You'll want to remove this eventually
		Config: &callee.Config{PassthroughMode: true},
	}
	if err := ret.Start(); err != nil {
		return nil, errors.Wrapf(err, "unable to start s2s client to %s", ret.ServiceName)
	}
	return ret, nil
}

func (s *service) twirpHooks() *twirp.ServerHooks {
	statsClient := s.BaseService.Statsd()
	tracker := twirpserviceslohook.TwirpSLOTracker{
		SLO:        timotyenapp.SLOTimotyenapp,
		DefaultSLO: time.Second,
		StatTracker: &statsdslo.StatTracker{
			Statter: statsClient.NewSubStatter("twirp.slo"),
		},
	}
	return twirp.ChainHooks(
		twirpstatsd.NewStatsdServerHooks(statsClient),
		tracker.ServerHook(),
	)
}

func (s *service) setup() error {
	var baseErr error
	s.BaseService, baseErr = stackcs4.New(stackcs4.Config{
		Team:           serviceTeam,
		Service:        serviceName,
		OptionalConfig: s.optionalConfig,
	})
	if baseErr != nil {
		return errors.Wrap(baseErr, "unable to init stacks")
	}
	s.BaseService.Logger().Log("user", os.Geteuid())
	if err := s.Config.Load(s.BaseService.Parameters()); err != nil {
		return errors.Wrap(err, "unable to load config parameters")
	}
	if err := s.Secrets.Load(s.BaseService.Secrets()); err != nil {
		return errors.Wrap(err, "uanble to load secrets")
	}
	if s.Secrets.AppSecret == "" {
		s.BaseService.Logger().Log("unable to find app secrets")
	}
	awsSession, err := session.NewSession(s.BaseService.AWSConfig())
	if err != nil {
		return errors.Wrap(err, "unable to make root aws session")
	}

	s2sClient, err := s.setupAuth()
	if err != nil {
		return errors.Wrap(err, "unable to create s2s client")
	}
	impl := &twirptimotyenapp.TimotyenappImpl{
		SecretKey:        s.Secrets.AppSecret,
		ItemGetTableName: s.Config.ItemGetTable,

		DynamoDB:          dynamodb.New(awsSession),
		DownstreamCircuit: s.BaseService.CircuitManager().MustCreateCircuit("downstream"),
		DynamoCircuit: s.BaseService.CircuitManager().MustCreateCircuit("dynamodb", circuit.Config{
			Execution: circuit.ExecutionConfig{
				Timeout: time.Second * 5,
			},
		}),
	}
	twirpServer := timotyenapp.NewTimotyenappServer(impl, s.twirpHooks())
	httpServer := httpapi.HTTPServer{
		HTTPServer: basehttpserver.HTTPServer{
			Config: &basehttpserver.Config{
				HealthcheckEndpoint: "/_health/" + serviceName,
				ListenAddr:          s.BaseService.Parameters().Str("listen_addr", "").Get(),
			},
			Log:      s.BaseService.Logger(),
			OnListen: s.OnListen,
		},
		TimotyenappServer: twirpServer,
		S2SClient:         s2sClient,
	}

	timotyenappClient := timotyenapp.NewTimotyenappProtobufClient(s.Config.TimotyenappHost, &http.Client{
		Transport: &http.Transport{
			DialContext: s.BaseService.DialContext(),
		},
	})
	s.runner.Log = s.BaseService.Logger()
	s.runner.Services = []servicerunner.Service{
		&httpServer,
		// TODO: Remove this service.  All it does is generate traffic
		&twirptimotyenapp.DoThingsService{
			Timotyenapp: timotyenappClient,
			Log:         s.BaseService.Logger(),
		},
	}
	s.runner.Closers = []io.Closer{
		s.BaseService,
	}
	// Verify the serivice is workable before we move forward
	if err := impl.HealthCheck(); err != nil {
		return err
	}
	s.BaseService.Logger().Log("I eventually want to revert this")
	if s2sClient != nil {
		s.runner.Closers = append(s.runner.Closers, s2sClient)
	}
	return nil
}

func (s *service) main() {
	if strings.Join(s.OsArgs, "") == "test" {
		fmt.Println("working", serviceName)
		s.osExit(0)
		return
	}
	if err := s.setup(); err != nil {
		s.BaseService.Logger().Log("err", err, "Unable to load initial setup")
		time.Sleep(time.Second)
		s.osExit(1)
		return
	}
	s.BaseService.Logger().Log("executing service timotyenapp!")
	if err := s.runner.Execute(); err != nil {
		s.BaseService.Logger().Log("err", err, "wait to end finished with an error")
		s.osExit(1)
		return
	}
	s.BaseService.Logger().Log("Finished main")
}

func main() {
	instance.main()
}
