package loglogrus

import (
	"fmt"
	"os"
)

var _ logrusAPI = &Log{}

// Log sends logrus messages to logger.  The default level is 'Panic'
type Log struct {
	Logger Logger
	Level  Level
}

// Debug logs at DebugLevel
func (l *Log) Debug(args ...interface{}) {
	l.levelPrint(DebugLevel, args...)
}

// Debugf logs at DebugLevel
func (l *Log) Debugf(format string, args ...interface{}) {
	l.levelF(DebugLevel, format, args...)
}

// Debugln logs at DebugLevel
func (l *Log) Debugln(args ...interface{}) {
	l.levelLn(DebugLevel, args...)
}

// Error logs at ErrorLevel
func (l *Log) Error(args ...interface{}) {
	l.levelPrint(ErrorLevel, args...)
}

// Errorf logs at ErrorLevel
func (l *Log) Errorf(format string, args ...interface{}) {
	l.levelF(ErrorLevel, format, args...)
}

// Errorln logs at ErrorLevel
func (l *Log) Errorln(args ...interface{}) {
	l.levelLn(ErrorLevel, args...)
}

// Fatal logs at FatalLevel and calls os.Exit
func (l *Log) Fatal(args ...interface{}) {
	l.levelPrint(FatalLevel, args...)
	if l.Level <= FatalLevel {
		os.Exit(1)
	}
}

// Fatalf logs at FatalLevel and calls os.Exit
func (l *Log) Fatalf(format string, args ...interface{}) {
	l.levelF(FatalLevel, format, args...)
	if l.Level <= FatalLevel {
		os.Exit(1)
	}
}

// Fatalln logs at FatalLevel and calls os.Exit
func (l *Log) Fatalln(args ...interface{}) {
	l.levelLn(FatalLevel, args...)
	if l.Level <= FatalLevel {
		os.Exit(1)
	}
}

// Info logs at InfoLevel
func (l *Log) Info(args ...interface{}) {
	l.levelPrint(InfoLevel, args...)
}

// Infof logs at InfoLevel
func (l *Log) Infof(format string, args ...interface{}) {
	l.levelF(InfoLevel, format, args...)
}

// Infoln logs at InfoLevel
func (l *Log) Infoln(args ...interface{}) {
	l.levelLn(InfoLevel, args...)
}

// Panic panics with args
func (l *Log) Panic(args ...interface{}) {
	panic(fmt.Sprint(args...))
}

// Panicf panics with args
func (l *Log) Panicf(format string, args ...interface{}) {
	panic(fmt.Sprintf(format, args...))
}

// Panicln panics with args
func (l *Log) Panicln(args ...interface{}) {
	panic(fmt.Sprintln(args...))
}

// Print logs at InfoLevel
func (l *Log) Print(args ...interface{}) {
	l.Info(args...)
}

// Printf logs at InfoLevel
func (l *Log) Printf(format string, args ...interface{}) {
	l.Infof(format, args...)
}

// Println logs at InfoLevel
func (l *Log) Println(args ...interface{}) {
	l.Infoln(args...)
}

// Warn logs at WarnLevel
func (l *Log) Warn(args ...interface{}) {
	l.levelPrint(WarnLevel, args...)
}

// Warnf logs at WarnLevel
func (l *Log) Warnf(format string, args ...interface{}) {
	l.levelF(WarnLevel, format, args...)
}

// Warnln at WarnLevel
func (l *Log) Warnln(args ...interface{}) {
	l.levelLn(WarnLevel, args...)
}

func (l *Log) levelPrint(lvl Level, args ...interface{}) {
	if !l.isLevel(lvl) {
		return
	}
	l.Logger.Log("level", lvl, fmt.Sprint(args...))
}

func (l *Log) levelLn(lvl Level, args ...interface{}) {
	if !l.isLevel(lvl) {
		return
	}
	l.Logger.Log("level", lvl, fmt.Sprintln(args...))
}

func (l *Log) levelF(lvl Level, format string, args ...interface{}) {
	if !l.isLevel(lvl) {
		return
	}
	l.Logger.Log("level", lvl, fmt.Sprintf(format, args...))
}

func (l *Log) isLevel(lvl Level) bool {
	return l.Level >= lvl
}

// Level controls if we ignore a log
type Level int32

const (
	// PanicLevel level, highest level of severity. Logs and then calls panic with the
	// message passed to Debug, Info, ...
	PanicLevel Level = iota
	// FatalLevel level. Logs and then calls `logger.Exit(1)`. It will exit even if the
	// logging level is set to Panic.
	FatalLevel
	// ErrorLevel level. Logs. Used for errors that should definitely be noted.
	// Commonly used for hooks to send errors to an error tracking service.
	ErrorLevel
	// WarnLevel level. Non-critical entries that deserve eyes.
	WarnLevel
	// InfoLevel level. General operational entries about what's going on inside the
	// application.
	InfoLevel
	// DebugLevel level. Usually only enabled when debugging. Very verbose logging.
	DebugLevel
)

// Logger is the root logger to send your messages to
type Logger interface {
	Log(keyvals ...interface{})
}

type logrusAPI interface {
	Debug(args ...interface{})
	Debugf(format string, args ...interface{})
	Debugln(args ...interface{})
	Error(args ...interface{})
	Errorf(format string, args ...interface{})
	Errorln(args ...interface{})
	Fatal(args ...interface{})
	Fatalf(format string, args ...interface{})
	Fatalln(args ...interface{})
	Info(args ...interface{})
	Infof(format string, args ...interface{})
	Infoln(args ...interface{})
	Panic(args ...interface{})
	Panicf(format string, args ...interface{})
	Panicln(args ...interface{})
	Print(args ...interface{})
	Printf(format string, args ...interface{})
	Println(args ...interface{})
	Warn(args ...interface{})
	Warnf(format string, args ...interface{})
	Warnln(args ...interface{})
}
