package twitchautoprof

import (
	"context"

	"code.justin.tv/hygienic/errors"
	"code.justin.tv/video/autoprof/profs3"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
)

// Logger logs values
type Logger interface {
	Log(vals ...interface{})
}

// Service configues autoprof
type Service struct {
	Config   *aws.Config
	S3Bucket string
	Logger   Logger

	collector profs3.Collector

	ctx    context.Context
	cancel context.CancelFunc
	done   chan struct{}
}

// Setup a autoprof service
func (s *Service) Setup() error {
	if s.S3Bucket == "" {
		return errors.New("expected S3Bucket to be non-empty")
	}

	s.ctx, s.cancel = context.WithCancel(context.Background())
	s.done = make(chan struct{})

	sess, err := session.NewSession(s.Config)
	if err != nil {
		return errors.Wrap(err, "new AWS session")
	}

	s.collector = profs3.Collector{
		S3:       s3.New(sess),
		S3Bucket: s.S3Bucket,
		OnError:  s.onError,
	}

	return nil
}

// Start a autoprof service
func (s *Service) Start() error {
	defer close(s.done)

	if err := s.collector.Run(s.ctx); err != nil && err != context.Canceled {
		return errors.Wrap(err, "autoprof collector fatal error")
	}

	return nil
}

// Close a autoprof service
func (s *Service) Close() error {
	s.cancel()
	<-s.done
	return nil
}

func (s *Service) onError(err error) error {
	s.log("err", err, "unexpected autoprof collector error")
	return nil
}

func (s *Service) log(vals ...interface{}) {
	if s.Logger != nil {
		s.Logger.Log(vals...)
	}
}
