package twirptimotyenservice

import (
	"context"
	"fmt"
	"math/big"
	"math/rand"
	"os"
	"sync"
	"time"

	"code.justin.tv/hygienic/errors"
	"code.justin.tv/hygienic/log"
	"code.justin.tv/timotyenorg/timotyenservice/proto/timotyenservice"
)

// DoThingsService just generates traffic so we can get interesting pprof profiles
type DoThingsService struct {
	Timotyenservice timotyenservice.Timotyenservice
	Log             log.Logger
	done            chan struct{}
}

// Setup should be called before start or close
func (d *DoThingsService) Setup() error {
	d.done = make(chan struct{})
	return nil
}

func (d *DoThingsService) showOffPrints() {
	if d.Log != nil && rand.Float64() < .2 {
		d.Log.Log("here is a message")
	}
	if d.Log != nil && rand.Float64() < .1 {
		d.Log.Log("err", errors.New("this error happens 10%"), "here is an error")
	}
	if d.Log != nil && rand.Float64() < .2 {
		d.Log.Log("err", errors.New("this error happens 20%"), "here is an error")
	}
}

func (d *DoThingsService) logic(ctx context.Context, i int, wg *sync.WaitGroup, delay time.Duration) {
	defer wg.Done()
	for {
		select {
		case <-d.done:
			return
		case <-time.After(time.Duration(rand.Int63n(delay.Nanoseconds()))):
		}
		d.showOffPrints()
		_, err := d.Timotyenservice.ItemGet(ctx, &timotyenservice.ItemGetRequest{
			Key: fmt.Sprintf("%d:%d", i, rand.Int()), // nolint
		})
		if err != nil && d.Log != nil {
			d.Log.Log("err", err)
		}
		// Add 50 ms of chug
		x := big.NewInt(10)
		end := time.Now().Add(50 * time.Millisecond)
		for time.Now().Before(end) {
			x = x.Add(x, big.NewInt(10))
		}
		if x.Int64() == 0 {
			panic("Want to use x so it doesn't get removed by smart compilers")
		}
	}
}

// Start doing random stuff to generate traffic
func (d *DoThingsService) Start() error {
	select {
	case <-d.done:
		return nil
	case <-time.After(time.Second * 1):
	}
	ctx, onCancel := context.WithCancel(context.Background())
	defer onCancel()
	go func() {
		select {
		case <-d.done:
			onCancel()
		case <-ctx.Done():
		}
	}()
	d.Log.Log("start")
	// Should average 1 req / sec
	const delay = time.Second * 10
	numGoroutines := 10

	if os.Getenv("ENVIRONMENT") == "development" {
		numGoroutines = 1
	}

	wg := sync.WaitGroup{}
	for i := 0; i < numGoroutines; i++ {
		wg.Add(1)
		go d.logic(ctx, i, &wg, delay)
	}
	wg.Wait()
	d.Log.Log("end")
	return nil
}

// Close signals start should end
func (d *DoThingsService) Close() error {
	close(d.done)
	return nil
}
