#!/bin/bash

export ORIGINAL_AWS_SHARED_CREDENTIALS_FILE="${ORIGINAL_AWS_SHARED_CREDENTIALS_FILE-${AWS_SHARED_CREDENTIALS_FILE-}}"
export AWS_SHARED_CREDENTIALS_FILE="$(abspath "${DIR}/../../credentials.ini")"

function root__credentials() {
  if [[ "${1-}" == "help" ]]; then
    echo "Runs credentials related commands"
    exit 0
  fi
  unset AWS_PROFILE
  export AWS_SHARED_CREDENTIALS_FILE=${ORIGINAL_AWS_SHARED_CREDENTIALS_FILE}
  if in_jenkins; then
    export AWS_PROFILE=${OPS_PIPELINE_PROFILE}
  fi
  process_build credentials__ $@
}

function credentials__builder_download() {
  if [[ "${1-}" == "help" ]]; then
    echo "aws-cli credentials to download builder credentials"
    exit 0
  fi
  # This cannot use the builder since obviously we are downloading the builder
  if in_jenkins; then
    assume_role --role-arn ${PIPELINE_RUNNING_ROLE} --role-session-name "role-from-jenkins-${BUILD_ID-}"
  else
    isengard_assume_role ${DEV_ACCOUNT_ID} ${DEV_ISENGARD_ROLE}
  fi

  cat <<EOF
{
  "AccessKeyId": "${AWS_ACCESS_KEY_ID}",
  "AccountId": "${DEV_ACCOUNT_ID}",
  "SecretAccessKey": "${AWS_SECRET_ACCESS_KEY}",
  "SessionToken": "${AWS_SESSION_TOKEN}",
  "Version": 1
}
EOF
}

function credentials__builder_upload() {
  if [[ "${1-}" == "help" ]]; then
    echo "aws-cli credentials to download builder credentials"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run credentials builder_upload $@
  fi
  if in_jenkins; then
    assume_role --role-arn ${PIPELINE_RUNNING_ROLE} --role-session-name "role-from-jenkins-${BUILD_ID-}"
  else
    isengard_assume_role ${OPS_PIPELINE_ACCOUNT_ID} ${OPS_PIPELINE_ROLE}
  fi

  cat <<EOF
{
  "AccessKeyId": "${AWS_ACCESS_KEY_ID}",
  "AccountId": "${DEV_ACCOUNT_ID}",
  "SecretAccessKey": "${AWS_SECRET_ACCESS_KEY}",
  "SessionToken": "${AWS_SESSION_TOKEN}",
  "Version": 1
}
EOF
}

# Assume a role from isengard.  $1 is the account ID and $2 is the role name
function isengard_assume_role() {
  if [ ! -f ~/.midway/cookie ]; then
    echo "midway cookie not found.  You may need to run mwinit"
    exit 1
  fi

  if ! credentials=$(isengard_get_credentials $@); then
    echo "${credentials}"
    exit 1
  fi
  export AWS_ACCESS_KEY_ID="$(echo "$credentials" | jq -r '.accessKeyId')";
  export AWS_SESSION_TOKEN="$(echo "$credentials" | jq -r '.sessionToken')";
  export AWS_SECRET_ACCESS_KEY="$(echo "$credentials" | jq -r '.secretAccessKey')";
}

function credentials__isengard_credentials() {
  if [[ "${1-}" == "help" ]]; then
    echo "Print isengard credentials for the aws-cli credential_process command"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run credentials isengard_credentials $@
  fi
  local account_id="" role="" assume=""
    while [[ "$#" -gt 0 ]]; do
    case "$1" in
      --account-id)
        account_id="$2"
        shift 2
        ;;
      --account-env)
        account_id="${!2}"
        shift 2
        ;;
      --role)
        role="$2"
        shift 2
        ;;
      --role-env)
        role="${!2}"
        shift 2
        ;;
      --assume)
        role="$2"
        shift 2
        ;;
      *)
        echo "Unrecognized option $1"
        exit 1
        ;;
    esac
  done
  require-argument --account-id "$account_id"
  require-argument --role "$role"
  if ! CREDS=$(isengard_get_credentials ${account_id} ${role}); then
    echo ${CREDS}
    exit 1
  fi
  echo "$CREDS" | jq -Mr '.["AccessKeyId"] = .accessKeyId | .["SecretAccessKey"] = .secretAccessKey | .["SessionToken"] = .sessionToken | .["Expiration"] = (.expiration / 1000 | todate) | del(.accessKeyId, .secretAccessKey, .sessionToken, .expiration) | .["Version"] = 1'
}

function credentials__deploy_credentials() {
  if [[ "${1-}" == "help" ]]; then
    echo "Print deployment credentials for the aws-cli commands"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run credentials deploy_credentials $@
  fi

  local account_id="" infra=""
    while [[ "$#" -gt 0 ]]; do
    case "$1" in
      --account-id)
        account_id="$2"
        shift 2
        ;;
      --account-env)
        account_id="${!2}"
        shift 2
        ;;
      --infra)
        infra="$2"
        shift 2
        ;;
      *)
        echo "Unrecognized option $1"
        exit 1
        ;;
    esac
  done
  require-argument --account-id "$account_id"
  require-argument --infra "$infra"
  if in_jenkins; then
    assume_role --role-arn ${PIPELINE_RUNNING_ROLE} --role-session-name "role-from-jenkins-${BUILD_ID-}"
  else
    isengard_assume_role ${account_id} ${DEV_ISENGARD_ROLE} # TODO: This may not be the right role.  Depends upon the
                                                            # account
  fi

  unset AWS_PROFILE
  local SESSION_NAME="push-${infra}-$(whoami)"
  local deploy_role="arn:aws:iam::${account_id}:role/${SERVICE}-${infra}-ops-deployer"
  assume_role --role-arn ${deploy_role} --role-session-name ${SESSION_NAME}
  cat <<EOF
{
  "AccessKeyId": "${AWS_ACCESS_KEY_ID}",
  "SecretAccessKey": "${AWS_SECRET_ACCESS_KEY}",
  "AccountId": "${account_id}",
  "SessionToken": "${AWS_SESSION_TOKEN}",
  "Version": 1
}
EOF
}

# Fetch credentials from isengard and populate ENV.  These creds are used by AWS SDK to authenticate.  Note that
# it uses a local file system cache to prevent spamming isengard.
# The location is ${ISENGARD_CACHE_DIR} (set above)
# $1 is the account ID and $2 is the role name
function isengard_get_credentials() {
  if [[ ! -f ~/.midway/cookie ]]; then
    echo "midway cookie not found.  You may need to run mwinit"
    exit 1
  fi

  local ACCOUNT_ID=${1-} ROLE=${2-}
  mkdir -p ${ISENGARD_CACHE_DIR}
  chmod 0700 "${ISENGARD_CACHE_DIR}"
  local ISENGARD_CACHE="${ISENGARD_CACHE_DIR}/${ACCOUNT_ID}-${ROLE}.json"
  # refresh cache at 10 minutes
  local REFRESH_AT=${ISENGARD_CACHE_TIME-600000}
  if [[ -f ${ISENGARD_CACHE} ]]; then
    local credentials=$(cat ${ISENGARD_CACHE})
    local EXPIRATION="$(echo "$credentials" | jq -r '.expiration')";
    local CURTIME=$(($(date +%s)*1000))
    local LEFTOVER=$(($EXPIRATION-$CURTIME))
    if (( LEFTOVER > REFRESH_AT )); then
      echo ${credentials}
      return 0
    fi
    # Cache bad.  Kill it with fire
    rm -f ${ISENGARD_CACHE}
  fi
  local credentials_and_code=$(curl --silent --write-out "HTTPSTATUS:%{http_code}" -k -b ~/.midway/cookie -c ~/.midway/cookie -L -X POST --header "X-Amz-Target: IsengardService.GetAssumeRoleCredentials" --header "Content-Encoding: amz-1.0" --header "Content-Type: application/json; charset=UTF-8" -d "{\"AWSAccountID\": \"$1\", \"IAMRoleName\":\"$2\"}" https://isengard-service.amazon.com);
  local HTTP_STATUS=$(echo "${credentials_and_code}" | tr -d '\n' | sed -e 's/.*HTTPSTATUS://')
  local HTTP_BODY=$(echo "${credentials_and_code}" | sed -e 's/HTTPSTATUS\:.*//g')
  if (( "200" != "${HTTP_STATUS}")); then
    echo "Invalid isengard status code: ${HTTP_STATUS}"
    echo "${HTTP_BODY}"
    exit 1
  fi
  local credentials=$(echo "${HTTP_BODY}" | jq -r '.AssumeRoleResult | fromjson | .credentials')
  echo ${credentials} > ${ISENGARD_CACHE}
  chmod 0600 "${ISENGARD_CACHE}"
  echo ${credentials}
}
