#!/bin/bash

# A place for deprecated functions

# Assume the right bootstrap role for an environment
function infra_assume_bootstrap_role() {
  INFRA="${1-}"
  SESSION_NAME="bootstrap-${INFRA}-$(whoami)"
  if [[ "${1-}" == "pipeline" ]]; then
    isengard_assume_role ${OPS_PIPELINE_ACCOUNT_ID} ${OPS_PIPELINE_ROLE}
    return
  fi
  if [[ "${1-}" == "staging" ]]; then
    isengard_assume_role ${DEV_ACCOUNT_ID} ${DEV_ISENGARD_ROLE}
    return
  fi
  if [[ "${1-}" == "personal" ]]; then
    isengard_assume_role ${DEV_ACCOUNT_ID} ${DEV_ISENGARD_ROLE}
    return
  fi
  if [[ "${1-}" == "production" ]]; then
    isengard_assume_role ${PROD_ACCOUNT_ID} ${PROD_ISENGARD_ROLE}
    return
  fi
  echo "Unable to assume role for infra ${INFRA}.  Is this a valid infra name?"
  exit 1
}

# Assume the deployment role inside an infrastructure
function infra_assume_infra_role() {
  INFRA="${1-}"
  SESSION_NAME="push-${INFRA}-$(whoami)"
  # If jenkins, assume pipeline first.  Otherwise try to assume the role directly
  infra_assume_pipeline_role

  # TODO: Abstract this better?

  if [[ "${1-}" == "personal" ]]; then
    isengard_assume_role ${DEV_ACCOUNT_ID} ${DEV_ISENGARD_ROLE}
    assume_role --role-arn $(personal_deploy_role) --role-session-name ${SESSION_NAME}
    return
  fi

  if [[ "${1-}" == "staging" ]]; then
    if in_jenkins; then
      assume_role --role-arn ${DEV_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    else
      isengard_assume_role ${DEV_ACCOUNT_ID} ${DEV_ISENGARD_ROLE}
      assume_role --role-arn ${DEV_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    fi
    return
  fi
  if [[ "${1-}" == "production" ]]; then
    if in_jenkins; then
      assume_role --role-arn ${PROD_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    else
      isengard_assume_role ${PROD_ACCOUNT_ID} ${PROD_ISENGARD_ROLE}
      assume_role --role-arn ${PROD_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    fi
    return
  fi
  if [[ "${1-}" == "canary" ]]; then
    if in_jenkins; then
      assume_role --role-arn ${PROD_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    else
      isengard_assume_role ${PROD_ACCOUNT_ID} ${PROD_ISENGARD_ROLE}
      assume_role --role-arn ${PROD_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    fi
    return
  fi
  echo "Unable to assume role for deployment ${INFRA}.  Is this a valid infra name?"
  exit 1
}

function infra_assume_pipeline_role() {
  # Note: only supported in jenkins so far
  if in_jenkins; then
    AWS_PROFILE=${OPS_PIPELINE_PROFILE} assume_role --role-arn ${PIPELINE_RUNNING_ROLE} --role-session-name "role-from-jenkins-${BUILD_ID-}"
  fi
}

function infra__deploy() {
  if [[ "${1-}" == "help" ]]; then
    echo "Deploy a service into an infrastructure (DEPRICATED)"
    exit 0
  fi
  INFRA="${1-}"
  if [[ -z ${INFRA} ]]; then
    echo "Pick an infra.  Valid infra to deploy: "
    echo
    print_valid_infra_deploy
    exit 1
  fi
  infra_assume_infra_role ${INFRA}
  make_run extra cloudformation -a ./cloudformation/deploy/${INFRA}.json
}

function infra__pipeline() {
  if [[ "${1-}" == "help" ]]; then
    echo "Update pipeline infrastructure (DEPRECATED)"
    exit 0
  fi
  CFEXECUTE_OPTIONS=""
  while getopts ":a" opt; do
    case ${opt} in
      a)
        CFEXECUTE_OPTIONS="${CFEXECUTE_OPTIONS} -a"
        ;;
      \?)
        echo "Invalid option: -$OPTARG" >&2
        return 1
        ;;
    esac
  done
  shift "$((OPTIND-1))"
  infra_assume_bootstrap_role pipeline

  make_run extra cloudformation ${CFEXECUTE_OPTIONS} ./cloudformation/service/service-pipeline.json
}

function infra__update() {
  # TODO: Add -a support here.
  if [[ "${1-}" == "help" ]]; then
    echo "Update infrastructure (DEPRECATED)"
    exit 0
  fi
  CFEXECUTE_OPTIONS=""
  while getopts ":a" opt; do
    case ${opt} in
      a)
        CFEXECUTE_OPTIONS="${CFEXECUTE_OPTIONS} -a"
        ;;
      \?)
        echo "Invalid option: -$OPTARG" >&2
        return 1
        ;;
    esac
  done
  shift "$((OPTIND-1))"

  INFRA="${1-}"
  if [[ -z ${INFRA} ]]; then
    echo "Pick an infra.  Valid infra to update: "
    echo
    print_valid_infra
    exit 1
  fi
  infra_assume_bootstrap_role ${INFRA}
  make_run extra cloudformation ${CFEXECUTE_OPTIONS} ./cloudformation/infra/${INFRA}.json
}

# Pull an image from ECR
function ecr_pull() {
  (
    REMOTE_TAG=${1-}
    DOCKER_CMD=$(aws ecr get-login --no-include-email --registry-ids ${OPS_PIPELINE_ACCOUNT_ID})
    eval ${DOCKER_CMD}

    DOCKER_CMD=$(aws ecr get-login --no-include-email --registry-ids ${DEV_ACCOUNT_ID})
    eval ${DOCKER_CMD}

    DOCKER_CMD=$(aws ecr get-login --no-include-email --registry-ids ${PROD_ACCOUNT_ID})
    eval ${DOCKER_CMD}
    docker pull ${REMOTE_TAG}
  )
}

function go__dep_check() {
  if [[ "${1-}" == "help" ]]; then
    echo "Verify dep project is in a good state (DEPRECATED)"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run go dep_check $@
  fi
  go mod verify
}

function extra__cloudformation() {
  if [[ "${1-}" == "help" ]]; then
    echo "Runs cloudformation command without assuming any roles"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run extra cloudformation $@
  fi
  CFEXECUTE_OPTIONS="-clean"
  AUTO=""
  while getopts ":a" opt; do
    case ${opt} in
      a)
        CFEXECUTE_OPTIONS="${CFEXECUTE_OPTIONS} -auto"
        AUTO="true"
        ;;
      \?)
        echo "Invalid option: -$OPTARG" >&2
        return 1
        ;;
    esac
  done
  shift "$((OPTIND-1))"

  STACK_LOC="${1-}"
  if [ -z ${STACK_LOC} ]; then
    echo "please specify a stack to update"
    exit 1
  fi
  if [[ ! -f "${STACK_LOC}" ]]; then
    echo "unable to find changeset file ${STACK_LOC}"
    exit 1
  fi
  CFEXECUTE_OPTIONS="${CFEXECUTE_OPTIONS} -changeset ${STACK_LOC}"

  if [[ "${DEBUG-}" = "true" ]]; then
    CFEXECUTE_OPTIONS="${CFEXECUTE_OPTIONS} -verbosity 2"
  fi
  if [ ! -z ${AUTO} ]; then
    # prep_term and wait_term are how you forward signals into bash child processes
    prep_term
    cfexecute ${CFEXECUTE_OPTIONS} &
    wait_term
  else
    cfexecute ${CFEXECUTE_OPTIONS}
  fi
}

function root__terraform() {
  if [[ "${1-}" == "help" ]]; then
    echo "Run terraform commands"
    exit 0
  fi
  process_build terraform__ $@
}

function terraform__apply() {
  if [[ "${1-}" == "help" ]]; then
    echo "Apply the ops terraform (DEPRECATED)"
    exit 0
  fi
  make_run twinit enable || true
  if [[ "${USE_BUILDER}" == "true" ]]; then
    USE_TWINIT=true make_exec builder run terraform apply $@
  fi
  (
    cd terraform
    infra_assume_bootstrap_role pipeline
    # Use -get-plugins=false to enforce that we are vendoring the plugins we need inside the builder
    terraform init -get-plugins=false
    GRAFANA_AUTH=$(ops_sandstorm grafana-key) PAGERDUTY_TOKEN=$(ops_sandstorm pagerduty) terraform apply
  )
}

function terraform__format() {
  if [[ "${1-}" == "help" ]]; then
    echo "Format all terraform files (DEPRECATED)"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run terraform format $@
  fi
  (
    cd terraform
    terraform fmt
  )
}

# It's ok to reuse the communications values
OPS_PREFIX=${OPS_PREFIX-search-di""scovery/servicecreation/development}

function root__sandstorm() {
  if [[ "${1-}" == "help" ]]; then
    echo "Fetch secrets from sandstorm"
    exit 0
  fi
  process_build sandstorm__ $@
}

function sandstorm__ops() {
  if [[ "${1-}" == "help" ]]; then
    echo "Fetch an ops secret from sandstorm"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run sandstorm ops $@
  fi
  if [[ -z "${1-}" ]]; then
    echo "Pass in a secret to fetch"
    exit 1
  fi
  ops_sandstorm $@
}

function ops_sandstorm() {
  (
    infra_assume_bootstrap_role pipeline
    sandstorm-cli get --role-arn ${OPS_SECRETS_ARN} ${OPS_PREFIX}/${1}
  )
}

# Combination of prep_term/handle_term/wait_term allow forwarding SIG from bash into another command
# See https://unix.stackexchange.com/questions/146756/forward-sigterm-to-child-in-bash for why we need this
# Forwarding signals allows us to forward a SIGKILl to running docker containers from our bash script
function prep_term() {
    unset term_child_pid
    unset term_kill_needed
    trap 'handle_term' TERM INT
}
function handle_term() {
    if [ "${term_child_pid}" ]; then
        kill -TERM "${term_child_pid}" 2>/dev/null
        # The child process will probably send some RPC to clean itself up.  Give it a tiny amount of time to actually
        # do that.
        sleep 2
        # Wait for the child process to finish killing itself
        wait ${term_child_pid}
    else
        term_kill_needed="yes"
    fi
}
function wait_term() {
    term_child_pid=$!
    if [ "${term_kill_needed-}" ]; then
        kill -TERM "${term_child_pid}" 2>/dev/null
    fi
    wait ${term_child_pid}
    trap - TERM INT
    wait ${term_child_pid}
}


function personal_deploy_role() {
  echo "arn:aws:iam::${DEV_ACCOUNT_ID}:role/${SERVICE}-personal${RUNNER}-ops-deployer"
}
