package seh1

import (
	"code.justin.tv/hygienic/metrics/metricsext"
	"math"

	"code.justin.tv/hygienic/metrics"
)

// bucketFactor is the configured bucket width.
// This should not be changed since it needs to match datastores that also use SEH1 histograms.
// What this comment means is that MWS is implemented in a way (sad face) that requires SEH1 histograms and REQUIRES
// this exact bucket factor for their API to make sense.
var defaultBucketFactor = math.Log(1 + 0.1)

// SEH1 is a sample aggregator that produces a sparse exponential histogram using an epsilon of 0.1.
type SEH1 struct {
	BucketFactor float64
	// Histogram is a sparse exponential histogram representing the samples that were aggregated into this Datum.
	// It can be used to estimate percentiles.
	// Since SEH1 does not work for the value zero and negative values, these are counted using SEH1ZeroBucket.
	histogram map[int32]int32

	// ZeroBucket is the bucket for counting zero values.
	zeroBucket int32
}

var _ metrics.Bucketer = &SEH1{}

func (h *SEH1) bucketFactor() float64 {
	if h.BucketFactor == 0 {
		return defaultBucketFactor
	}
	return h.BucketFactor
}

// Buckets returns all the bucketed values for this histogram
func (h *SEH1) Buckets() []metrics.Bucket {
	ret := make([]metrics.Bucket, 0, len(h.histogram))
	// Set the individual values
	for bucketVal, sampleCount := range h.histogram {
		ret = append(ret, metrics.Bucket{
			Start: inverseBucket(bucketVal, h.bucketFactor()),
			End:   inverseBucket(bucketVal + 1, h.bucketFactor()),
			Count: sampleCount,
		})
	}
	if h.zeroBucket > 0 {
		ret = append(ret, metrics.Bucket{
			Start: 0,
			End:   0,
			Count: h.zeroBucket,
		})
	}
	return ret
}

// Observe adds a sample to the histogram.
func (h *SEH1) Observe(value float64) {
	if h.histogram == nil {
		h.histogram = make(map[int32]int32)
	}
	if value > 0 {
		bucket := computeBucket(value, h.bucketFactor())
		h.histogram[bucket]++
	} else {
		h.zeroBucket++
	}
}

func inverseBucket(bucketIndex int32, bucketFactor float64) float64 {
	return math.Pow(math.E, float64(bucketIndex)*bucketFactor)
}

// helper to compute the bucket for a sample value
func computeBucket(v float64, bucketFactor float64) int32 {
	return int32(math.Floor(math.Log(v) / bucketFactor))
}

// SEHAggregationConstructor returns the same SEH rolling aggregation for each time series
func SEHRollingAggregation(_ *metrics.TimeSeries) metrics.Aggregator {
	return &metricsext.RollingAggregation{
		AggregatorFactory: func() metrics.ValueAggregator {
			return &metricsext.LocklessValueAggregator {
				Bucketer: &SEH1{},
			}
		},
	}
}
