package statsdslo

import (
	"fmt"
	"strings"

	"code.justin.tv/hygienic/twirpserviceslohook"
)

// StatTracker implements the stat tracker interface for twirpserviceslohook, but sends metrics to statsd
type StatTracker struct {
	// How to sanitize method and service names.  By default, allows [a-zA-Z0-9_]*
	Sanitizer func(string) string `json:"-"`
	// Statter is something that can increment statsd metrics. Matches the normal Go statsd interface
	Statter Statter `json:"-"`
	// Rate to sample metrics.  By default is .2
	SampleRate float32
	// OnStatsdError is called when Inc fails.  By default, the error is ignored
	OnStatsdError func(error) `json:"-"`
}

// Statter is similar to the statsd inc interface
type Statter interface {
	Inc(metric string, val int64, rate float32) error
}

func sanitize(s string) string {
	return strings.Map(sanitizeRune, s)
}

func sanitizeRune(r rune) rune {
	switch {
	case 'a' <= r && r <= 'z':
		return r
	case '0' <= r && r <= '9':
		return r
	case 'A' <= r && r <= 'Z':
		return r
	default:
		return '_'
	}
}

var _ twirpserviceslohook.StatTracker = &StatTracker{}

func (s *StatTracker) sampleRate() float32 {
	if s.SampleRate == 0 {
		return .2
	}
	return s.SampleRate
}

func (s *StatTracker) sanitize(in string) string {
	if s.Sanitizer == nil {
		return sanitize(in)
	}
	return s.Sanitizer(in)
}

func (s *StatTracker) common(serviceName string, methodName string, suffix string) {
	statName := fmt.Sprintf("%s.%s.%s", serviceName, methodName, suffix)
	if err := s.Statter.Inc(statName, 1, s.sampleRate()); err != nil && s.OnStatsdError != nil {
		s.OnStatsdError(err)
	}
}

// Passed will increment a 'passed' statsd metric
func (s *StatTracker) Passed(serviceName string, methodName string) {
	s.common(serviceName, methodName, "passed")
}

// Failed will increment a 'failed' statsd metric
func (s *StatTracker) Failed(serviceName string, methodName string) {
	s.common(serviceName, methodName, "failed")
}
