package stacks5

import (
	"context"
	"expvar"
	"net"
	"os"
	"strings"
	"sync"

	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/errors"
	"code.justin.tv/hygienic/expvar2"
	"code.justin.tv/hygienic/gometrics"
	"code.justin.tv/hygienic/log"
	"code.justin.tv/hygienic/sandyconf"
	"code.justin.tv/hygienic/servicerunner"
	"code.justin.tv/hygienic/twitchautoprof"
	"code.justin.tv/hygienic/twitchbaseservice/v4"
	"code.justin.tv/hygienic/twitchbaseservice/v4/internal"
	"code.justin.tv/hygienic/twitchcircuit"
	"code.justin.tv/hygienic/twitchdebugservice"
	"code.justin.tv/hygienic/twitchstatsd"
	"code.justin.tv/hygienic/twitchstdoutlogger"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/cep21/circuit"
)

// DevEnvironment is a special const that means you're running locally from a mac and need a SOCKS proxy
const DevEnvironment = twitchbaseservice.DevEnvironment

var _ twitchbaseservice.BaseService = &Stack{}
var _ twitchbaseservice.WithExpvar = &Stack{}

// Stack is a s5 stack.  SSM/StatsD/Sandstorm/etc
type Stack struct {
	secrets        *distconf.Distconf
	LoggerService  twitchstdoutlogger.Service
	CircuitService twitchcircuit.Service
	DebugService   twitchdebugservice.Service
	StatsService   twitchstatsd.Service
	serviceRunner  servicerunner.ServiceRunner
	params         *distconf.Distconf
	dialContext    func(ctx context.Context, network, addr string) (net.Conn, error)

	config                    Config
	backgroundExecuteErrors   chan error
	once                      sync.Once
	cachedEnvVariablesInUpper map[string]string
}

// Config is all required and configures how to setup your stack
type Config struct {
	Team           string
	Service        string
	OptionalConfig OptionalConfig
}

func (s *Stack) setupEnvCache() {
	allEnviron := os.Environ()
	s.cachedEnvVariablesInUpper = make(map[string]string, len(allEnviron))
	for _, originalEnvironKV := range os.Environ() {
		originalEnvironSplit := strings.SplitN(originalEnvironKV, "=", 2)
		if len(originalEnvironSplit) == 2 {
			s.cachedEnvVariablesInUpper[mutateEnvironKey(originalEnvironSplit[0])] = originalEnvironSplit[1]
		}
	}
}

func mutateEnvironKey(key string) string {
	return strings.Replace(strings.ToUpper(key), "-", "_", -1)
}

func (s *Stack) getEnv(key string) string {
	s.once.Do(s.setupEnvCache)
	return s.cachedEnvVariablesInUpper[mutateEnvironKey(key)]
}

func (s *Stack) environment() string {
	ret := s.config.OptionalConfig.Environment
	if ret == "" {
		ret = s.getEnv("environment")
	}
	return ret
}

func (s *Stack) fallbackEnvironment() string {
	if s.config.OptionalConfig.FallbackEnvironment != nil {
		return *s.config.OptionalConfig.FallbackEnvironment
	}
	defaultFallbacks := map[string]string{
		"development": "staging",
		"canary":      "production",
	}
	for thisEnv, fallbackEnv := range defaultFallbacks {
		if s.environment() == thisEnv {
			return fallbackEnv
		}
	}
	return ""
}

type OptionalConfig = twitchbaseservice.OptionalConfig

func (s *Stack) findSandstormARN() string {
	return s.Parameters().Str("sandstorm_arn", "").Get()
}

func filterInvalidReaders(readers []distconf.Reader) ([]distconf.Reader, error) {
	var retErr error
	ret := make([]distconf.Reader, 0, len(readers))
	for _, r := range readers {
		if _, err := r.Get("_"); err != nil {
			retErr = err
		} else {
			ret = append(ret, r)
		}
	}
	if len(ret) == 0 {
		return nil, errors.Wrap(retErr, "no readers are valid")
	}
	return ret, nil
}

func (s *Stack) makeSecrets() (*distconf.Distconf, error) {
	readers := make([]distconf.Reader, 0, 3)
	readers = append(readers, s.config.OptionalConfig.DefaultReaders...)
	readers = append(readers, &distconf.Env{
		OsGetenv: s.getEnv,
	})
	if sandstormARN := s.findSandstormARN(); sandstormARN != "" {
		sandstormReaders := make([]distconf.Reader, 0, 2)
		m := sandyconf.ManagerConstructor{
			AwsConfig: s.AWSConfig(),
		}
		mngr, err := m.Manager(sandstormARN)
		if err != nil {
			return nil, errors.Wrap(err, "uanble to make sandstorm manager")
		}
		sc := sandyconf.Sandyconf{
			Team:        s.config.Team,
			Service:     s.config.Service,
			Environment: s.environment(),
			Manager:     mngr,
		}
		sandstormReaders = append(sandstormReaders, &sc)
		if s.fallbackEnvironment() != "" {
			scFallback := sandyconf.Sandyconf{
				Team:        s.config.Team,
				Service:     s.config.Service,
				Environment: s.fallbackEnvironment(),
				Manager:     mngr,
			}
			sandstormReaders = append(sandstormReaders, &scFallback)
		}
		var filterErr error
		sandstormReaders, filterErr = filterInvalidReaders(sandstormReaders)
		if filterErr != nil {
			return nil, errors.Wrapf(filterErr, "unable to create sandstorm secret reader for %s", sandstormARN)
		}
		readers = append(readers, sandstormReaders...)
	}
	return &distconf.Distconf{
		Readers: readers,
		Logger: func(key string, err error, msg string) {
			s.Logger().Log("key", key, "err", err, "msg", msg)
		},
	}, nil
}

// New creates a stack or returns an error if it cannot be created
func New(config Config) (*Stack, error) {
	ret := &Stack{
		config: config,
		DebugService: twitchdebugservice.Service{
			Expvar2: expvar2.Handler{
				Exported: make(map[string]expvar.Var, 10),
			},
		},
		StatsService: twitchstatsd.Service{
			Team:    config.Team,
			Service: config.Service,
		},
	}
	ret.params = &distconf.Distconf{
		Readers: []distconf.Reader{
			&distconf.Env{
				OsGetenv: ret.getEnv,
			},
		},
		Logger: func(key string, err error, msg string) {
			ret.Logger().Log("key", key, "err", err, "msg", msg)
		},
	}
	ret.params.Readers = append(ret.params.Readers, config.OptionalConfig.DefaultReaders...)
	ret.dialContext = ret.createDialContext()
	ret.LoggerService.UseLogfmt = ret.environment() == DevEnvironment
	ret.StatsService.Environment = ret.environment()

	if ret.environment() == DevEnvironment {
		ret.StatsService.Statsd = &statsd.NoopClient{}
	}

	if ret.environment() == DevEnvironment && !config.OptionalConfig.DisableSocksVerification {
		if err := ret.verifyDialContext(); err != nil {
			return ret, errors.Wrap(err, "unable to verify socks proxy")
		}
	}

	ret.StatsService.Log = ret.Logger()
	if err := ret.LoggerService.Setup(); err != nil {
		return ret, errors.Wrap(err, "unable to setup logging service")
	}

	if err := ret.verifyAWSConn(); err != nil {
		return ret, err
	}

	if err := ret.StatsService.Config.Load(ret.Parameters()); err != nil {
		return ret, errors.Wrap(err, "unable to setup statsd config")
	}
	if err := ret.StatsService.Setup(); err != nil {
		return ret, errors.Wrap(err, "unable to setup statsd service")
	}

	var secretsErr error
	ret.secrets, secretsErr = ret.makeSecrets()
	if secretsErr != nil {
		return ret, errors.Wrap(secretsErr, "unable to setup secrets")
	}

	ret.CircuitService = twitchcircuit.Service{
		Statsd: ret.StatsService.Statsd.NewSubStatter(""),
		Log:    ret.Logger(),
		Config: ret.Parameters(),
	}
	ret.setupExpvars()
	ret.setupAutoprof()
	ret.setupGometrics()
	ret.serviceRunner.Log = ret.Logger()
	ret.serviceRunner.Append(&ret.LoggerService, &ret.CircuitService, &ret.StatsService)
	if !config.OptionalConfig.IgnoreDebugService {
		ret.serviceRunner.Append(&ret.DebugService)
	}
	ret.backgroundExecuteErrors = make(chan error, 1)
	retErr := ret.serviceRunner.ExecuteInBackground(func(err error) {
		ret.backgroundExecuteErrors <- err
	})
	return ret, errors.Wrap(retErr, "unable to execute stack in background")
}

func (s *Stack) setupExpvars() {
	internal.SetupExpvar(s)
}

func (s *Stack) setupAutoprof() {
	bucket := s.Parameters().Str("autoprof_bucket", "").Get()
	if bucket != "" {
		autoprofService := twitchautoprof.Service{
			Logger:   s.Logger(),
			Config:   s.AWSConfig(),
			S3Bucket: bucket,
		}
		s.serviceRunner.Append(&autoprofService)
	}
}

func (s *Stack) setupGometrics() {
	gometricsService := gometrics.Service{
		Logger:  s.Logger(),
		Statter: s.StatsService.Statsd.NewSubStatter(""),
	}
	s.serviceRunner.Append(&gometricsService)
}

// Parameters configure non secret parts of your service
func (s *Stack) Parameters() *distconf.Distconf {
	return s.params
}

// Secrets are for secret information that must be protected, like API keys
func (s *Stack) Secrets() *distconf.Distconf {
	return s.secrets
}

// Logger is how you should log
func (s *Stack) Logger() log.Logger {
	if s.config.OptionalConfig.ForcedLogger != nil {
		return s.config.OptionalConfig.ForcedLogger
	}
	return &s.LoggerService
}

// CircuitManager is for outbound requests
func (s *Stack) CircuitManager() *circuit.Manager {
	return &s.CircuitService.Circuit
}

// Statsd is for basemetrics
func (s *Stack) Statsd() statsd.SubStatter {
	return s.StatsService.Statsd.NewSubStatter("")
}

// DialContext should be used for all outbound connections.
func (s *Stack) DialContext() func(ctx context.Context, network, addr string) (net.Conn, error) {
	return s.dialContext
}

// AWSConfig should be used as the base for all AWS sessions
func (s *Stack) AWSConfig() *aws.Config {
	return &aws.Config{}
}

// ExpvarHandler should eventually be exposed so we can profile information
func (s *Stack) ExpvarHandler() *expvar2.Handler {
	return &s.DebugService.Expvar2
}

// Close when finished with this stack
func (s *Stack) Close() error {
	if err := s.serviceRunner.Close(); err != nil {
		return err
	}
	return <-s.backgroundExecuteErrors
}

func (s *Stack) createDialContext() func(ctx context.Context, network, addr string) (net.Conn, error) {
	return internal.CreateDialContext(s, s.environment(), s.config.OptionalConfig)
}

func (s *Stack) verifyAWSConn() error {
	return internal.VerifyAWSConn(s)
}

// verifyDialContext checks if an instance metadata request succeeds using DialContext()
func (s *Stack) verifyDialContext() error {
	return internal.VerifyDialContext(s)
}
