package twitchdebugservice

import (
	"context"
	"net"
	"net/http"
	"net/http/pprof"
	"strings"
	"time"

	"code.justin.tv/hygienic/expvar2"
)

// Config controls how to setup the service
type Config struct {
	ListenAddr string
}

func (c Config) listenAddr() string {
	if c.ListenAddr == "" {
		return ":6060"
	}
	return c.ListenAddr
}

// Service starts a 6060 debug service
type Service struct {
	Config        Config
	Expvar2       expvar2.Handler
	Mux           *http.ServeMux
	server        http.Server
	debugListener net.Listener
}

// Setup starts the listener
func (s *Service) Setup() error {
	if s.Mux == nil {
		s.Mux = http.NewServeMux()
	}
	s.server = http.Server{
		Handler: s.Mux,
	}

	s.Mux.Handle("/debug/vars", &s.Expvar2)
	s.Mux.Handle("/debug/pprof/", http.HandlerFunc(pprof.Index))
	s.Mux.Handle("/debug/pprof/cmdline", http.HandlerFunc(pprof.Cmdline))
	s.Mux.Handle("/debug/pprof/profile", http.HandlerFunc(pprof.Profile))
	s.Mux.Handle("/debug/pprof/symbol", http.HandlerFunc(pprof.Symbol))
	s.Mux.Handle("/debug/pprof/trace", http.HandlerFunc(pprof.Trace))

	var err error
	s.debugListener, err = retriedListen(context.Background(), s.Config.listenAddr(), time.Second*30, time.Second)
	return err
}

// Start listens for requests
func (s *Service) Start() error {
	return s.server.Serve(s.debugListener)
}

// Close ends the server
func (s *Service) Close() error {
	return s.server.Close()
}

// retriedListen continues to attempt to listen on a port if the port is taken, until the port is free.  This is important
// because our deploy process restarts a service a bit too quickly during deploy and may think it just is a failed push
func retriedListen(ctx context.Context, addr string, retryTime time.Duration, sleepTime time.Duration) (net.Listener, error) {
	endTime := time.Now().Add(retryTime)
	var listener net.Listener
	var lastError error
	for {
		listener, lastError = net.Listen("tcp", addr)
		if lastError == nil {
			return listener, nil
		}
		if endTime.Before(time.Now()) {
			return nil, lastError
		}
		if strings.Contains(lastError.Error(), "address already in use") {
			select {
			case <-time.After(sleepTime):
			case <-ctx.Done():
				return nil, ctx.Err()
			}
			continue
		}
		return nil, lastError
	}
}
