package twitchstdoutlogger

import (
	"fmt"
	"io"
	"os"
	"sync"
	"sync/atomic"

	"code.justin.tv/hygienic/log/fmtlogger"

	"code.justin.tv/hygienic/errors"
	"code.justin.tv/hygienic/log"
)

// Service allows async stdout logging.  Services can use JSON for easy parsing while local runs can use
// logfmt since it's easier to read
type Service struct {
	Output      io.Writer
	UseLogfmt   bool
	asyncLogger log.ChannelLogger
	intoLogger  log.Logger
	countLogger log.Counter
	closeSignal chan struct{}
	once        sync.Once
	running     int32
}

var _ log.Logger = &Service{}

func (s *Service) setup() {
	s.closeSignal = make(chan struct{})
	if s.Output == nil {
		s.Output = os.Stdout
	}
	if s.UseLogfmt {
		s.intoLogger = &log.ErrorLogLogger{
			RootLogger: &fmtlogger.LogfmtLogger{
				Out: s.Output,
				MissingValueKey: log.Msg,
			},
			ErrHandler: &s.countLogger,
		}
	} else {
		s.intoLogger = &log.ErrorLogLogger{
			RootLogger: &log.JSONLogger{
				Out: s.Output,
				MissingValueKey: log.Msg,
			},
			ErrHandler: &s.countLogger,
		}
	}
	s.asyncLogger = log.ChannelLogger{
		Out:    make(chan []interface{}, 1024),
		OnFull: &s.countLogger,
	}
}

// Setup the service so it can run
func (s *Service) Setup() error {
	s.once.Do(s.setup)
	return nil
}

// MissedLogs returns the count of async logs that had to be skipped
func (s *Service) MissedLogs() int64 {
	return atomic.LoadInt64(&s.countLogger.Count)
}

// Log a msg
func (s *Service) Log(keyvals ...interface{}) {
	s.once.Do(s.setup)
	keyvals = wrapKeyvalsWithStackTrace(keyvals)
	if atomic.LoadInt32(&s.running) == 1 {
		s.asyncLogger.Log(keyvals...)
		return
	}
	// Otherwise skip async logging
	s.intoLogger.Log(keyvals...)
}

// Start waits for close
func (s *Service) Start() error {
	s.once.Do(s.setup)
	atomic.StoreInt32(&s.running, 1)
	defer atomic.StoreInt32(&s.running, 0)
	defer s.blockingDrain()
	for {
		select {
		case <-s.closeSignal:
			return nil
		case msgs := <-s.asyncLogger.Out:
			s.intoLogger.Log(msgs...)
		}
	}
}

func (s *Service) blockingDrain() {
	for {
		select {
		case msgs := <-s.asyncLogger.Out:
			s.intoLogger.Log(msgs...)
		default:
			return
		}
	}
}

// Close ends the async logger
func (s *Service) Close() error {
	s.once.Do(s.setup)
	close(s.closeSignal)
	return nil
}

func wrapKeyvalsWithStackTrace(keyvals []interface{}) []interface{} {
	for idx := 0; idx+1 < len(keyvals); idx += 2 {
		if isErrorKey(keyvals[idx]) {
			if err, ok := keyvals[idx+1].(error); ok {
				keyvals[idx+1] = wrapErrIfNeeded(err)
			}
		}
	}
	return keyvals
}

func isErrorKey(k interface{}) bool {
	if k == nil {
		return false
	}
	switch x := k.(type) {
	case string:
		return x == "err"
	case fmt.Stringer:
		return x != nil && x.String() == "err"
	default:
		return false
	}
}

func wrapErrIfNeeded(err error) error {
	type tracedError1 interface {
		StackTrace() []uintptr
	}
	type tracedError2 interface {
		Stack() []uintptr
	}
	if _, ok := err.(tracedError1); ok {
		return err
	}
	if _, ok := err.(tracedError2); ok {
		return err
	}
	return errors.Wrap(err, "")
}
