﻿using System.Collections.Generic;
using UnityEditor;
using UnityEngine;

namespace TwitchInGames {
    [InitializeOnLoad]
    public class SwitchPlatform {

        private static Dictionary<BuildTarget, string[]> pluginPaths = new Dictionary<BuildTarget, string[]>() {
            { BuildTarget.StandaloneWindows64,  new[] { "Assets/Plugins/x86_64/Unity.dll" ,"Assets/Plugins/x86_64/UnityWin32.dll" } },
            { BuildTarget.iOS,                  new[] { "Assets/Plugins/iOS/Unity.dll", "Assets/Plugins/iOS/libUnityiOS.a" } },
            { BuildTarget.Android,              new[] { "Assets/Plugins/Android/Unity.dll" } }
        };

        private static Dictionary<BuildTarget, PluginImporter[]> plugins = new Dictionary<BuildTarget, PluginImporter[]>(3);

        static SwitchPlatform() {
            BuildTarget target = EditorUserBuildSettings.activeBuildTarget;

            string availablePlugins = "";
            foreach(var kvp in pluginPaths) {
                int length = kvp.Value.Length;
                PluginImporter[] pluginImporters = new PluginImporter[length];
                bool isTarget = kvp.Key == target ? true : false;
                try {
                    for(int i = 0; i < length; i++) {
                        pluginImporters[i] = AssetImporter.GetAtPath(kvp.Value[i]) as PluginImporter;
                        // TODO: Adding new plugin folder or files and reload fails to detect UnityWin32.dll and TwigWebview.jar
                        if(pluginImporters[i] == null)
                            throw new MissingReferenceException(string.Format("[TwitchInGames] Missing file for platform {0}: {1}", kvp.Key.ToString(), kvp.Value[i]));
                        pluginImporters[i].SetCompatibleWithEditor(isTarget);
                    }
                } catch(MissingReferenceException e) {
                    if(isTarget)
                        Debug.LogError(e.Message);
                    else
                        Debug.LogWarning(e.Message);
                    continue;
                }

                plugins.Add(kvp.Key, pluginImporters);
                availablePlugins = availablePlugins + " " + kvp.Key.ToString();
            }

            switch(target) {
            case BuildTarget.StandaloneWindows64:
                EditorUserBuildSettings.SwitchActiveBuildTarget(BuildTargetGroup.Standalone, target);
                break;
            case BuildTarget.iOS:
                EditorUserBuildSettings.SwitchActiveBuildTarget(BuildTargetGroup.iOS, target);
                break;
            case BuildTarget.Android:
                EditorUserBuildSettings.SwitchActiveBuildTarget(BuildTargetGroup.Android, target);
                break;
            }

            Debug.Log("[TwitchInGames] Current build target platform: " + target.ToString() + "\nPlugins imported for platforms: " + availablePlugins);
        }

        [MenuItem("TwitchInGames/Switch Platform/Windows 64")]
        public static void SwitchWindows64() {
            Switch(BuildTarget.StandaloneWindows64);
        }

        [MenuItem("TwitchInGames/Switch Platform/iOS")]
        public static void SwitchiOS() {
            Switch(BuildTarget.iOS);
        }

        [MenuItem("TwitchInGames/Switch Platform/Android")]
        public static void SwitchAndroid() {
            Switch(BuildTarget.Android);
        }

        public static void Switch(BuildTarget target) {
            foreach(var kvp in plugins) {
                if(kvp.Key == target) {
                    foreach(var plugin in kvp.Value) {
                        plugin.SetCompatibleWithEditor(true);
                    }
                } else {
                    foreach(var plugin in kvp.Value) {
                        plugin.SetCompatibleWithEditor(false);
                    }
                }
            }

            switch(target) {
            case BuildTarget.StandaloneWindows64:
                EditorUserBuildSettings.SwitchActiveBuildTarget(BuildTargetGroup.Standalone, target);
                break;
            case BuildTarget.iOS:
                EditorUserBuildSettings.SwitchActiveBuildTarget(BuildTargetGroup.iOS, target);
                break;
            case BuildTarget.Android:
                EditorUserBuildSettings.SwitchActiveBuildTarget(BuildTargetGroup.Android, target);
                break;
            }
        }
    }
}
