import { requestIsengardCredentials, UnauthenticatedError, UnknownIsengardError, OffNetworkUnauthenticatedError, ResponseParseError } from './request-isengard-credentials';
import * as aws from 'aws-sdk';

test('succeeds', async () => {
  const credentials = await requestIsengardCredentials({
    accountID: 'account-id',
    roleName: 'admin',
    requestCreds: () => successResponse(),
    authenticate: () => {
      throw new Error("Authenticate should not have been called.");
    },
  });

  expect(credentials).toEqual(dummyCredentials)
});

test('authenticates and then succeeds', async () => {
  const authenticate = jest.fn();
  const credentials = await requestIsengardCredentials({
    accountID: 'account-id',
    roleName: 'admin',
    requestCreds: () => {
      if (authenticate.mock.calls.length) {
        return unauthenticatedResponse();
      } else {
        return successResponse();
      }
    },
    authenticate,
  });

  expect(credentials).toEqual(dummyCredentials)
});

test('never authenticates and throws', async () => {
  const authenticate = jest.fn();
  const request = requestIsengardCredentials({
    accountID: 'account-id',
    roleName: 'admin',
    requestCreds: () => unauthenticatedResponse(),
    authenticate,
  });

  await expect(request).rejects.toEqual(new UnauthenticatedError());
  expect(authenticate).toHaveBeenCalledTimes(1);
});

test('unknown response error throws', async () => {
  const request = requestIsengardCredentials({
    accountID: 'account-id',
    roleName: 'admin',
    requestCreds: () => unknownErrorResponse(),
    authenticate: () => {
      throw new Error("Authenticate should not have been called.");
    },
  });

  await expect(request).rejects.toBeInstanceOf(UnknownIsengardError);
});

test('off network unauthenticated response succeeds after authentication', async () => {
  const authenticate = jest.fn();
  const credentials = await requestIsengardCredentials({
    accountID: 'account-id',
    roleName: 'admin',
    requestCreds: () => {
      if (authenticate.mock.calls.length) {
        return offNetworkUnauthenticatedResponse();
      } else {
        return successResponse();
      }
    },
    authenticate,
  });

  await expect(credentials).toEqual(dummyCredentials)
});

test('invalid JSON throws', async () => {
  const request = requestIsengardCredentials({
    accountID: 'account-id',
    roleName: 'admin',
    requestCreds: () => Promise.resolve('invalid'),
    authenticate: () => {
      throw new Error("Authenticate should not have been called.");
    },
  });

  try {
    await request
    throw new Error("Expected call to throw");
  } catch(e) {
    expect(e).toBeInstanceOf(ResponseParseError);
    expect(e).toHaveProperty('message', 'Failed to parse response from Isengard: invalid');
  }
});

const dummyCredentials = new aws.Credentials({
  accessKeyId: 'access-key-id',
  secretAccessKey: 'secret-access-key',
})

function successResponse() {
  const response = JSON.stringify({
    AssumeRoleResult: JSON.stringify({
      credentials: {
        accessKeyId: 'access-key-id',
        secretAccessKey: 'secret-access-key',
      }
    })
  });

  return Promise.resolve(response);
}

function unauthenticatedResponse() {
  const response = JSON.stringify({
    status: 'error',
    message: 'Unauthenticated'
  });

  return Promise.resolve(response);
}

function unknownErrorResponse() {
  const response = JSON.stringify({
    status: 'error',
    message: 'something we do not expect',
    description: 'more information about the error',
  });

  return Promise.resolve(response);
}

function offNetworkUnauthenticatedResponse() {
  const response = JSON.stringify({
    compliance_valid: false,
  });

  return Promise.resolve(response);
}
