import * as aws from 'aws-sdk';
interface RequestIsengardCredentialsParams {
  accountID: string;
  roleName: string;
  requestCreds: (accountID: string, roleName: string) => Promise<string>;
  authenticate: () => void;
}

export async function requestIsengardCredentials(params: RequestIsengardCredentialsParams, retryCount: number = 0): Promise<aws.Credentials> {
  const stdout = await params.requestCreds(params.accountID, params.roleName);

  let credentials;
  try {
    credentials = credentialsFromIsengardResponse(stdout);
  } catch(e) {
    if (
      !retryCount &&
      e instanceof UnauthenticatedError ||
      e instanceof OffNetworkUnauthenticatedError
    ) {
      console.log('🔒 \x1b[33mYou need to login with mwinit.\x1b[0m');
      params.authenticate();
      return requestIsengardCredentials(params, ++retryCount);
    } else {
      throw e;
    }
  }

  return new aws.Credentials(credentials);
}

function credentialsFromIsengardResponse(payload: string) {
  let rpcResponse;
  try {
    rpcResponse = JSON.parse(payload);
  } catch(e) {
    throw new ResponseParseError(payload);
  }

  // Authentication errors from known networks have good error messages.
  if (rpcResponse.status === 'error') {
    if (rpcResponse.message === 'Unauthenticated') {
      throw new UnauthenticatedError();
    }

    throw new UnknownIsengardError([rpcResponse.message, rpcResponse.desc].join(': '));
  }

  // Off network, unauthenticated errors to midway are different.
  if (rpcResponse.compliance_valid === false) {
    throw new OffNetworkUnauthenticatedError(rpcResponse.message);
  }

  if (!rpcResponse.AssumeRoleResult) {
    throw new Error(`Expected the Isengard response to have key 'AssumeRoleResult': ${payload}`);
  }

  let assumeRoleResult;
  try {
    assumeRoleResult = JSON.parse(rpcResponse.AssumeRoleResult);
  } catch(e) {
    throw new Error(`Failed to parse Isengard AssumeRoleResult: ${rpcResponse.AssumeRoleResult}`)
  }

  const credentials = assumeRoleResult.credentials;
  if (!credentials) {
    throw new Error(`Expected AssumeRoleResult to have 'credentials': ${assumeRoleResult}`);
  }

  return credentials;
}

export class UnauthenticatedError extends Error {
  name = 'unauthenticated';
  constructor(message?: string) {
    super(message || 'Not authenticated to Isengard');
  }
}

export class UnknownIsengardError extends Error {
  name = 'unknown';
  constructor(message?: string) {
    super(message || 'Unknown Isengard error');
  }
}

export class OffNetworkUnauthenticatedError extends Error {
  name = 'off-network-unauthenticated';
  constructor(message: string) {
    super(`Off network authentication error, trying running mwinit to resolve: ${message}`);
  }
}

export class ResponseParseError extends Error {
  name = 'response-parse';
  constructor(payload: string) {
    super(`Failed to parse response from Isengard: ${payload}`);
  }
}
