#!/usr/bin/env node
import * as child_process from "child_process";
import * as fs from "fs";
import { URL } from "url";
import * as yargs from "yargs";

export function uploadDependencies(
  bucket: string,
  airflowVersion: string,
  twitchAirflowComponentsVersion: string
) {
  // Clean build dir.
  child_process.execSync("rm -rf build && mkdir -p build");

  console.log("Downloading twitch-airflow-components wheel.");
  child_process.execSync(
    `pip download --no-deps twitch-airflow-components==${twitchAirflowComponentsVersion} -d build`
  );

  console.log("Creating plugins.zip archive.");
  child_process.execSync(
    `zip -r -j build/plugins.zip build/twitch_airflow_components*`
  );

  console.log("Writing requirements.txt file.");
  const requirementsBody = `/usr/local/airflow/plugins/twitch_airflow_components-${twitchAirflowComponentsVersion}-py3-none-any.whl\ncloudpickle==2.0.0`;
  fs.writeFileSync("build/requirements.txt", requirementsBody);

  console.log("Uploading files to S3.");
  child_process.execSync(
    `aws s3 cp build/plugins.zip ${new URL("plugins.zip", bucket).href}`
  );

  child_process.execSync(
    `aws s3 cp build/requirements.txt ${
      new URL("requirements.txt", bucket).href
    }`
  );
}

const argv = yargs(process.argv.slice(2))
  .options({
    "s3-bucket": {
      describe: "MWAA S3 bucket to install dependencies in.",
      demandOption: true,
      type: "string",
    },
    "airflow-version": {
      describe: "Airflow version of your MWAA cluster.",
      demandOption: true,
      type: "string",
    },
    "twitch-airflow-components-version": {
      describe: "Version if Twitch Airflow Components to install.",
      demandOption: true,
      type: "string",
    },
  })
  .help().argv;

uploadDependencies(
  argv["s3-bucket"],
  argv["airflow-version"],
  argv["twitch-airflow-components-version"]
);
