import { CfnInitConfig } from ".";
import iam = require("@aws-cdk/aws-iam");

/**
 * Add custom metric reporting from EC2 instances with the CloudWatch agent.
 *
 * Adapted from https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/solutions/AmazonCloudWatchAgent/inline/amazon_linux.template
 */
export function cloudWatchAgentConfig(): CfnInitConfig {
  const cwAgentControlPath = `/opt/aws/amazon-cloudwatch-agent/bin/amazon-cloudwatch-agent-ctl`;
  const cwAgentConfigPath = `/opt/aws/amazon-cloudwatch-agent/etc/amazon-cloudwatch-agent.json`;

  return {
    onLaunch: {
      name: "install-cloudwatch-agent",
      commands: [
        {
          name: "install-agent",
          command: "rpm -Uvh https://s3.amazonaws.com/amazoncloudwatch-agent/amazon_linux/amd64/latest/amazon-cloudwatch-agent.rpm",
        },
      ],
    },
    always: {
      name: "run-cloudwatch-agent",
      files: [
        {
          path: cwAgentConfigPath,
          content: {
            metrics: {
              namespace: "Flink",
              append_dimensions: {
                InstanceId: "${!aws:InstanceId}",
              },
              metrics_collected: {
                cpu: {
                  measurement: [
                    "cpu_usage_user",
                    "cpu_usage_system"
                  ]
                },
                mem: {
                  measurement: ["mem_used_percent"],
                }
              },
              aggregation_dimensions: [["InstanceId"], []],
            },
          },
        },
      ],
      commands: [
        {
          name: "stop-service",
          command: `${cwAgentControlPath} -a stop`
        },
        {
          name: "start-service",
          command: `${cwAgentControlPath} -a fetch-config -m ec2 -c file:${cwAgentConfigPath} -s`
        }
      ]
    },
    addAccess: role => {
      role.addManagedPolicy(
        iam.ManagedPolicy.fromAwsManagedPolicyName(
          "CloudWatchAgentServerPolicy"
        )
      );
    }
  };
}
