# Github Consumer

This repository contains CDK constructs for working with the Twitch
Github Enterprise instance.

## Branch Stacks

This package facilitates per-branch CloudFormation stacks. You might want to
use this module if you're deploying a new stack for every branch in GitHub
Enterprise.

### Usage

```ts
import * as cdk from '@aws-cdk/core';
import { branchstacks } from '@twitch-cdk/github-consumer';

const app = new cdk.App();
const env = { account: "141429874596", region: "us-west-2" };

// Create a Repo object which holds some common configuration to
// access and identify your GitHub repo.
const repo = new branchstacks.Repo({
  name: 'chemtrail',
  owner: 'ce-analytics',
  auth: cdk.SecretValue.secretsManager('github', { jsonField: 'oauth' }),
});

// Setup the Infrastructure stack that looks for Cfn stacks to destroy. You
// would probably instantiate Cleaner in its own stack rather than in the app
// file.
const branchstacksInfra = new cdk.Stack(app, 'BranchStacksCleaner', { env });
new branchstacks.Cleaner(branchstacksInfra, 'Cleaner', { repo }),

// You need some means to get the current branch name. You could
// use process.env or CDK's app context.
const currentBranch = repo.branch(app.node.tryGetContext('branch'));

// Use currentBranch.nameForStack to apply a suffix to the branch name. This
// will ensure that there are no stack naming conflicts.
const serviceStack = new MyServiceStack(app, currentBranch.nameForStack('MyService'), { env });

// Applying tags to the stack will cause the cleaner to delete the stack when
// the branch is gone.
currentBranch.tagAsDependent(serviceStack);
```

## Testing

You can test your package by using `npm link` to symlink code in this
project into another.

```
npm run build
npm link

cd path/to/project/repo
npm link "@twitch-cdk/github-consumer"
```

This will add the package to the project's node_modules directory and you can
import it. You can continue to work on the project and rebuild using `npm run
build` to have changes reflected in the project repository.

## Publishing

This package is published to our internal npm registry that is powered by
Artifactory.

Follow the instructions
[here](https://wiki.twitch.com/display/DTA/Artifactory%3A+User+Guides) for
retrieving your password.

`npm.internal.justin.tv` requires being connected to the Twitch VPN or using
`twinit`.

### Configure NPM

```bash
npm config set registry https://npm.internal.justin.tv/
npm login # use the password that was generated above
```
