import cdk = require('@aws-cdk/core');
import crypto = require('crypto');

interface RepoProps {
  /** 
   * This is the organization part of the github repo path, although the github
   * API calls it "owner". For instance, in the 'aws/aws-cdk' repository the
   * owner is 'aws'.
   */
  owner: string;

  /**
   * This is just the repo name part of a github repo path. For instance, in
   * the 'aws/aws-cdk' repository, the repo is 'aws-cdk'.
   */
  name: string; 

  /**
   * This is a Github OAuth value. You should store this value in AWS SecretsManager
   * and provide that cdk.SecretValue here to keep it hidden.
   */
  auth: cdk.SecretValue;

  /**
   * This is the URL pointing to github.com or a GHE instance.
   * 
   * @default https://git.xarth.tv/api/v3
   */
  url?: string;
}

export class Repo {
  public readonly owner: string;
  public readonly name: string;
  public readonly auth: cdk.SecretValue;
  public readonly url: string;
  public readonly ignoredBranches: string;

  constructor(props: RepoProps) {
    this.owner = props.owner;
    this.name = props.name;
    this.auth = props.auth;
    this.url = props.url ?? 'https://git.xarth.tv/api/v3';
  }

  /**
   * Create a branch object with utilities to help name and tag
   * your branch stacks.
   */
  public branch(branchName: string) {
    return new Branch(this, branchName);
  }
}

class Branch {
  private readonly repo: Repo;
  private readonly name: string;

  constructor(repo: Repo, name: string) {
    this.repo = repo;
    this.name = name;
  }

  /**
   * Safely append a string to a stack name while keeping the characters and the
   * length within AWS limits. Because the branch name is normalized and could
   * therefore cause a collision, append a hash to the end of the stack name.
   */
  public nameForStack(base: string) {
    const sanitizedBranch = this.name.replace(/[^-a-zA-Z0-9]/g, '-');
    const stackName = `${base}-${sanitizedBranch}`;
    const stackNameWithRoomForHash = stackName.substring(0, 128 - 6);
    const hash = crypto.createHash('md5').update(this.name).digest('hex').substring(0, 5);

    return `${stackNameWithRoomForHash}-${hash}`;
  }

  public tagAsDependent(stack: cdk.Stack) {
    const tags = cdk.Tags.of(stack);
    tags.add('github-owner', this.repo.owner);
    tags.add('github-repo', this.repo.name);
    tags.add('github-branch', this.name);
  }
}
