import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as cwactions from '@aws-cdk/aws-cloudwatch-actions';
import * as iam from '@aws-cdk/aws-iam';
import * as kinesis from '@aws-cdk/aws-kinesis';
import * as sns from '@aws-cdk/aws-sns';
import * as core from '@aws-cdk/core';

const DATA_INFRA_ACCOUNT = new iam.AccountPrincipal('673385534282');
const DATA_INFRA_SNS_TOPIC_ARN = 'arn:aws:sns:us-west-2:673385534282:spade-downstream-prod-alert';

interface SpadeKinesisStreamProps {
  /** The number of Kinesis shards to use for the stream */
  shardCount: number;

  /** 
   * The unique name of the stream. Data infra requires this name to begin with
   * `spade-downstream-prod` and recommends including the account name and
   * region in the name (e.g. `spade-downstream-prod-twitch-ce-analytics-prod-us-west-2-my-specific-stream`)
   */
  streamName: string;

  /**
   * This construct pages the DataInfra team when they send too much or too
   * little data to the Kinesis stream. Use this property to add additional
   * notifications for these data alarms.
   * 
   * @default []
   */
  additionalAlarmTopics?: sns.ITopic[]

  /**
   * We usually don't want to trigger alarms for development environments. Use
   * this field to enable or disarm the alarms.
   */
  alarmsEnabled: boolean;
}

export interface IKinesisStream {
  streamName: string;
  grantRead(grantable: iam.IGrantable): iam.Grant; 
}

/**
 * This is a stream setup with Data Infra team's recommendations.
 * It allows the data infra account to put spade data into the 
 * stream.
 */
export class KinesisStream extends core.Construct implements IKinesisStream {
  private stream: kinesis.Stream;

  constructor(scope: core.Construct, id: string, props: SpadeKinesisStreamProps) {
    super(scope, id);

    const writerRole = new iam.Role(this, 'Writer', {
      roleName: `${props.streamName}-putter`,
      assumedBy: DATA_INFRA_ACCOUNT,
    });

    this.stream = new kinesis.Stream(this, 'Resource', {
      streamName: props.streamName,
      shardCount: props.shardCount, 
      retentionPeriod: core.Duration.hours(48),
    });

    this.stream.grantWrite(writerRole);

    const dataInfraAlertTopic = sns.Topic.fromTopicArn(this, 'DataInfraAlertTopic', DATA_INFRA_SNS_TOPIC_ARN);
    const alertTopics = [dataInfraAlertTopic, ...(props.additionalAlarmTopics ?? [])];

    const incomingBytesMetric = new cloudwatch.Metric({
      namespace: 'AWS/Kinesis',
      statistic: 'Sum',
      metricName: 'IncomingBytes',
      dimensions: {
        StreamName: this.stream.streamName,
      },
    });

    this.createAlarm(incomingBytesMetric, 'TooFewBytes', {
      actionsEnabled: props.alarmsEnabled,
      alarmName: `${this.streamName} Too Few Bytes`,
      alarmDescription: `Fewer than 1% of max bytes sent to ${this.stream.streamName}`,
      comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
      period: core.Duration.minutes(1),
      evaluationPeriods: 2,
      threshold: 0.01 * props.shardCount * 1_000_000 * 60,
      treatMissingData: cloudwatch.TreatMissingData.BREACHING,
      alertTopics,
    });

    this.createAlarm(incomingBytesMetric, 'TooManyBytes', {
      actionsEnabled: props.alarmsEnabled,
      alarmName: `${this.stream.streamName} Too Many Bytes`,
      alarmDescription: `Too many bytes sent to ${this.stream.streamName}`,
      comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
      period: core.Duration.minutes(1),
      evaluationPeriods: 2,
      threshold: 0.85 * props.shardCount * 1_000_000 * 60,
      treatMissingData: cloudwatch.TreatMissingData.NOT_BREACHING,
      alertTopics,
    });

    const incomingRecordsMetric = new cloudwatch.Metric({
      namespace: 'AWS/Kinesis',
      statistic: 'Sum',
      metricName: 'IncomingRecords',
      dimensions: {
        StreamName: this.stream.streamName,
      }
    });

    this.createAlarm(incomingRecordsMetric, 'TooManyRecords', {
      actionsEnabled: props.alarmsEnabled,
      alarmName: `${this.stream.streamName} Too Many Records`,
      alarmDescription: `Too many records sent to ${this.stream.streamName}`,
      comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
      period: core.Duration.minutes(1),
      evaluationPeriods: 2,
      threshold: 0.85 * props.shardCount * 1_000 * 60,
      treatMissingData: cloudwatch.TreatMissingData.NOT_BREACHING,
      alertTopics,
    });
  }

  public grantRead(grantable: iam.IGrantable): iam.Grant {
    this.stream.grant(grantable, 'kinesis:DescribeStream')
    return this.stream.grantRead(grantable);
  }

  public get streamName() {
    return this.stream.streamName;
  }

  private createAlarm(metric: cloudwatch.Metric, name: string, options: cloudwatch.CreateAlarmOptions & { alertTopics: sns.ITopic[] }) {
    const alarm = metric.createAlarm(this, name, options);
    const snsActions = options.alertTopics.map(t => new cwactions.SnsAction(t));
    alarm.addAlarmAction(...snsActions);
    alarm.addOkAction(...snsActions);
    alarm.addInsufficientDataAction(...snsActions);
  }
}
