# Tahoe Tap

Connect a Redshift cluster to Tahoe.

## Getting Started

Install the dependency:

```
npm i @twitch-cdk/tahoe-tap
```

Normal caveats for peerDependencies for using CDK libraries apply. See [the
package.json](./package.json) file for a list of peerDependencies that you
should add to your app.

Create your Redshift cluster using CDK's [Redshift
module](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-redshift-readme.html) and add a `tap.Connection` to connect your Redshift cluster to Tahoe..

```ts
import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as redshift from '@aws-cdk/aws-redshift';
import * as s3 from '@aws-cdk/aws-s3';

import * as tap from '@twitch-cdk/tahoe-tap';

interface TahoeTapStackProps extends cdk.StackProps {
  vpc: ec2.IVpc;
}

class MyTapStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: TahoeTapStackProps) {
    super(scope, id, props);

    const tapName = 'my-tahoe-tap'

    const cluster = new redshift.Cluster(this, 'Cluster', {
      clusterName: tapName,
      vpc: props.vpc,
      masterUser: {
        masterUsername: 'tahoeadmin',
      },
      defaultDatabaseName: 'product',
    });
    cluster.connections.allowFrom(ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(5439));

    new tap.Connection(this, 'Connection', {
      bindleId: 'amzn1.bindle.resource.ABCDEfghij1234567890',
      cluster,
      purpose: "My tap's raison d'être",
      tapName,
      // Choose from a list of views on https://taproom.di.twitch.a2z.com/
      views: ["spade.invariant_ping"],
      vpc: props.vpc,
    });
  }
}
```

The names 'tahoeadmin' and 'product' are not required, but they follow the
convention of other Tahoe taps.


## Library Development

### Unit Tests

We have TypeScript and Python unit tests. To run the typescript tests:

To run all the tests use

```
make test
```

```
npx jest
```

To run the Python tests:

```
python -m unittest discover lib/handler -p '*_test.py'
```

### Integration Testing

To create a personal stack for testing, add your user ID to a .env file.

```
echo PERSONAL_STACK=$(whoami) > .env
```

Then run cdk command to diff and deploy.

```
npx cdk diff # see what is changing
npx cdk deploy 'Tahoe*' # deploy the test stack
```
