import * as ec2 from '@aws-cdk/aws-ec2';
import * as redshift from '@aws-cdk/aws-redshift';
import * as cdk from '@aws-cdk/core';
import * as tap from './';
import { EventRules } from './event-rules';
import { TapRoles } from './roles';
import { SubscriptionList } from './subscription-list';

interface ConnectionProps {
  /**
   * The friendly tap name that will appear in TapRoom.
   *
   * @example my-service
   */
  tapName: string;

  /**
   * An Amazon bindle ID from https://bindles.amazon.com.
   *
   * @example amzn1.bindle.resource.ABCDEfghij1234567890
   */
  bindleId: string;

  /**
   * A description indicating the tap's purpose.
   *
   * @example Powers the creator dashboards
   */
  purpose: string;

  /**
   * A list of views to subscribe to.
   *
   * @example ["spade.invariant_ping"]
   */
  views: string[];

  /**
   * A Redshift cluster.
   */
  cluster: redshift.Cluster;

  /**
   * The VPC for the tap's endpoint service.
   */
  vpc: ec2.IVpc;

  /**
   * The Tahoe stage to use for API calls.
   * 
   * @default TahoeProd
   */
  tahoeStage?: tap.TahoeStage;
  /**
   * Subnets to create NLB in.
   *
   * @defaults Vpc default strategy.
   */
  subnets?: ec2.SubnetSelection;
}

export class Connection extends cdk.Construct {
  constructor(scope: cdk.Construct, id: string, props: ConnectionProps) {
    super(scope, id);

    const endpointService = new tap.EndpointService(this, 'TahoeEndpoint', {
      cluster: props.cluster,
      vpc: props.vpc,
      principals: [cdk.Stack.of(this).account],
      tahoeStage: props.tahoeStage,
      subnets: props.subnets,
    });

    const secret = props.cluster.secret;
    if (!secret) {
      throw new Error('Redshift cluster must have a secret');
    }
    const roles = new TapRoles(this, 'TapRoles', {
      tahoeStage: props.tahoeStage,
      secret,
    });

    const reg = new tap.Registration(this, 'TahoeRegistration', {
      tapName: props.tapName,
      bindleId: props.bindleId,
      purpose: props.purpose,
      cluster: props.cluster,
      roles,
      endpointServiceName: endpointService.serviceName,
      tahoeStage: props.tahoeStage,
    });

    new EventRules(this, 'LocalViewEventRule', {
      tahoeStage: props.tahoeStage,
      tapId: reg.tapId,
      adminRoleArn: roles.adminRole.roleArn
    });

    new SubscriptionList(this, 'SubscriptionList', {
      tapId: reg.tapId,
      views: props.views,
    });
  }
}
