import * as redshift from '@aws-cdk/aws-redshift';
import * as cdk from '@aws-cdk/core';
import { TahoeStage } from '.';
import { ResourceHandler } from './resource-handler';
import { TapRoles } from './roles';

export interface RegistrationProps {
  /**
   * A humanized name for your tap. This will help you find your tap in Taproom.
   * 
   * @example my-service
   */
  tapName: string;
  cluster: redshift.Cluster,

  /**
   * An Amazon bindle ID from https://bindles.amazon.com.
   * 
   * @example amzn1.bindle.resource.ABCDEfghij1234567890
   */
  bindleId: string;
  purpose: string;
  roles: TapRoles;
  endpointServiceName: string;
  tahoeStage?: TahoeStage;
}

export class Registration extends cdk.Construct {
  public readonly tapId: string;

  constructor(scope: cdk.Construct, id: string, props: RegistrationProps) {
    super(scope, id);

    validateProps(props);
    const handler = new ResourceHandler(this, 'RegistrationHandler', {
      tahoeStage: props.tahoeStage,
    });

    const secretName = props.cluster.secret?.secretArn;
    if (!secretName) {
      throw new Error('Missing secret name');
    }

    const registration = new cdk.CustomResource(this, 'Resource', {
      serviceToken: handler.serviceToken,
      properties: {
        CustomResource: 'Registration',
        TapName: props.tapName,
        ClusterId: props.cluster.clusterName,
        BindleId: props.bindleId,
        Purpose: props.purpose,
        RedshiftSecretArn: secretName,
        TapAdminRoleArn: props.roles.adminRole.roleArn,
        TapDataRoleArn: props.roles.dataRole.roleArn,
        VpcEndpointService: props.endpointServiceName,
        RedshiftClusterId: props.cluster.clusterName,
      },
    });
    this.tapId = registration.getAttString('TapId')
  }
}

export interface ValidatedProps {
  tapName: string;
  bindleId: string;
}

const bindleValidationString = '^amzn1\.bindle\.resource\.[a-zA-Z0-9]+$';
const bindleRegexp = new RegExp(bindleValidationString);
const tapNameValidationString = /^[a-z][a-z0-9-]{1,46}[a-z0-9]$/;
const tapNameRegexp = new RegExp(tapNameValidationString);

export function validateProps(props: ValidatedProps) {
  if (!bindleRegexp.test(props.bindleId)) {
    throw new Error(`bindleId ${props.bindleId} must match ${bindleValidationString}`);
  }

  if (!tapNameRegexp.test(props.tapName)) {
    throw new Error(`tapName ${props.tapName} must match ${tapNameValidationString}`);
  }
}
