package main

import (
	"context"
	"encoding/csv"
	"flag"
	"fmt"
	"log"
	"os"
	"strings"
	"time"

	"github.com/satori/go.uuid"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/twitch-events/gea/lib/geaclient"
)

const (
	outputDirectory = "./output/"

	layout1 = "1/2/2006 3:04 PM MST"
	layout2 = "1/2/06 3:04 PM MST"
)

var (
	csvInputFileName = flag.String("i", "", "(Required) Filename of the target .csv to be converted to Twitch Events")
	utilityMode      = flag.String("mode", "", "(Required) Either [local], [staging], [production], or [dryrun] determines whether to fire against localhost, staging, production, or nothing respectively")

	utilityModeLocal      = "local"
	utilityModeStaging    = "staging"
	utilityModeProduction = "production"
	utilityModeDryRun     = "dryrun"

	csvTitle       = "Title"
	csvDescription = "Description"
	csvStartTime   = "Start Time"
	csvEndTime     = "End Time"
	csvTimeZone    = "Timezone"
	csvChannelName = "Channel"
	csvGameName    = "Game"
	csvParentEvent = "Parent Event"
	csvEventImage  = "Event Image URL"
	csvEventID     = "Event ID"

	csvTitleIndex       = -1
	csvDescriptionIndex = -1
	csvStartTimeIndex   = -1
	csvEndTimeIndex     = -1
	csvTimeZoneIndex    = -1
	csvChannelNameIndex = -1
	csvGameNameIndex    = -1
	csvParentEventIndex = -1
	csvEventImageIndex  = -1
	csvEventIDIndex     = -1

	csvColumnNames = make([]string, 0)

	geaHostURLMap = map[string]string{
		utilityModeLocal:      "http://localhost:8080/",
		utilityModeStaging:    "http://events-staging.internal.justin.tv/",
		utilityModeProduction: "http://events-production.internal.justin.tv/",
		utilityModeDryRun:     "",
	}
	geaHostTarget string

	channelNameToIDMap = map[string]string{
		"riotgames":       "36029255",  // riotgames
		"overwatchleague": "137512364", // overwatchleague
		// "riotgames": "40529315", // bern (for testing)
	}

	gameNameToIDMap = map[string]string{
		"League of Legends": "21779",
		"Overwatch":         "488552",
	}

	imageURLToIDMap = map[string]string{
		// NA LCS Event Collection Image
		"https://lolstatic-a.akamaihd.net/esports-assets/production/league/na-lcs-g63ljv52.png": "gea:abbf50ab-6889-41f8-84dd-151502344dd9",
		// EU LCS Event Collection Image
		"https://lolstatic-a.akamaihd.net/esports-assets/production/league/eu-lcs-egx86wrt.png": "gea:02fbbb31-8797-4355-b56b-272376c27460",
		// NA LCS Event Leaf Image
		"https://i.imgur.com/OSfv0QB.png": "gea:abbf50ab-6889-41f8-84dd-151502344dd9",
		// EU LCS Event Leaf Image
		"https://i.ytimg.com/vi/AnPwizdeaQY/maxresdefault_live.jpg": "gea:02fbbb31-8797-4355-b56b-272376c27460",
	}

	timezoneToIDMap = map[string]string{
		"PST": "America/Los_Angeles",
		"PDT": "America/Los_Angeles",
		"GMT": "Etc/GMT",
	}

	eventNameToIDMap = make(map[string]string, 0)

	eventsClient geaclient.Client
)

func main() {
	flag.Parse()
	validateFlags()

	var err error
	if geaHostTarget == utilityModeDryRun {
		fmt.Println("THIS IS A DRY RUN, which means events will not actually be created")
		time.Sleep(time.Second * 4)
	} else {
		eventsClient, err = geaclient.NewClient(
			twitchclient.ClientConf{
				Host: geaHostTarget,
			},
		)
		if err != nil {
			log.Fatal("unable to start gea client")
		}
		fmt.Println("starting geaclient pointed at", geaHostTarget, "..giving you a few seconds to reconsider ;)")
		time.Sleep(time.Second * 4)
	}

	csvFile, err := os.Open(*csvInputFileName)
	if err != nil {
		log.Fatal("invalid csv input filename=", err)
	}
	defer csvFile.Close()

	csvReader := csv.NewReader(csvFile)
	err = setupReader(csvReader)
	if err != nil {
		log.Fatal("error in csv reader setup=", err)
	}
	csvEntries, err := csvReader.ReadAll()
	if err != nil {
		log.Fatal("error reading csv entries=", err)
	}
	result, err := parseEntries(csvEntries)
	if err != nil {
		writeEntriesToCSV(result, false)
		log.Fatal("WRITE FAILED=", err)
	} else {
		writeEntriesToCSV(result, true)
	}
}

func validateFlags() {
	if *csvInputFileName == "" {
		flag.PrintDefaults()
		log.Fatal("Error parsing input flags. Please ensure all required flags are included in script invocation!")
	}

	if *utilityMode == utilityModeDryRun {
		geaHostTarget = utilityModeDryRun
	} else if *utilityMode == utilityModeLocal {
		geaHostTarget = geaHostURLMap[utilityModeLocal]
	} else if *utilityMode == utilityModeStaging {
		geaHostTarget = geaHostURLMap[utilityModeStaging]
	} else if *utilityMode == utilityModeProduction {
		geaHostTarget = geaHostURLMap[utilityModeProduction]
	} else {
		flag.PrintDefaults()
		log.Fatal("Error parsing input flags. Please ensure all required flags are included in script invocation!")
	}
}

func setupReader(reader *csv.Reader) error {
	var err error
	csvColumnNames, err = reader.Read()
	if err != nil {
		return err
	}
	for index, columnName := range csvColumnNames {
		if columnName == csvTitle {
			csvTitleIndex = index
		} else if columnName == csvDescription {
			csvDescriptionIndex = index
		} else if columnName == csvStartTime {
			csvStartTimeIndex = index
		} else if columnName == csvEndTime {
			csvEndTimeIndex = index
		} else if columnName == csvTimeZone {
			csvTimeZoneIndex = index
		} else if columnName == csvChannelName {
			csvChannelNameIndex = index
		} else if columnName == csvGameName {
			csvGameNameIndex = index
		} else if columnName == csvParentEvent {
			csvParentEventIndex = index
		} else if columnName == csvEventImage {
			csvEventImageIndex = index
		} else if columnName == csvEventID {
			csvEventIDIndex = index
		}
	}
	return nil
}

func parseEntries(entries [][]string) ([][]string, error) {
	for index, entry := range entries {
		if entry[csvParentEventIndex] == "" {
			// Container Event
			fmt.Printf("processing entry:\n%+v\n\n", entry)
			id := entry[csvEventIDIndex]
			title := entry[csvTitleIndex]
			description := entry[csvDescriptionIndex]
			channelID, ok := channelNameToIDMap[entry[csvChannelNameIndex]]
			if !ok {
				return entries, fmt.Errorf("could not find corresponding channel ID to %+v", entry[csvChannelNameIndex])
			}
			imageID := entry[csvEventImageIndex]
			if !strings.HasPrefix(imageID, "gea:") {
				imageID, ok = imageURLToIDMap[entry[csvEventImageIndex]]
				if !ok {
					return entries, fmt.Errorf("could not find corresponding image ID to %+v", entry[csvEventImageIndex])
				}
			}
			timezoneID, ok := timezoneToIDMap[entry[csvTimeZoneIndex]]
			if !ok {
				return entries, fmt.Errorf("could not find corresponding timezone ID to %+v", entry[csvTimeZoneIndex])
			}

			event := Event{
				ID:          id,
				Title:       title,
				Description: description,
				ChannelID:   channelID,
				ImageID:     imageID,
				TimeZoneID:  timezoneID,
			}

			var newID string
			var err error
			if geaHostTarget == utilityModeDryRun {
				newID = uuid.NewV4().String()
			} else {
				newID, err = generateEventCollection(event)
				if err != nil {
					return entries, fmt.Errorf("could not generate event %+v=%+v", event.Title, err)
				}
			}
			if id == "" {
				entry[csvEventIDIndex] = newID
				entries[index] = entry
			}
			eventNameToIDMap[title] = entry[csvEventIDIndex]
		} else {
			// Leaf Event
			fmt.Printf("processing entry:\n%+v\n\n", entry)
			id := entry[csvEventIDIndex]
			title := entry[csvTitleIndex]
			description := entry[csvDescriptionIndex]
			startAt, err := convertToTimestamp(entry[csvStartTimeIndex], entry[csvTimeZoneIndex])
			if err != nil {
				return entries, fmt.Errorf("unable to convert %+v to timestamp=%+v", entry[csvStartTimeIndex]+" "+entry[csvTimeZoneIndex], err)
			}
			endAt, err := convertToTimestamp(entry[csvEndTimeIndex], entry[csvTimeZoneIndex])
			if err != nil {
				return entries, fmt.Errorf("unable to convert %+v to timestamp=%+v", entry[csvStartTimeIndex]+" "+entry[csvTimeZoneIndex], err)
			}
			channelID, ok := channelNameToIDMap[entry[csvChannelNameIndex]]
			if !ok {
				return entries, fmt.Errorf("could not find corresponding channel ID to %+v", entry[csvChannelNameIndex])
			}
			gameID, ok := gameNameToIDMap[entry[csvGameNameIndex]]
			if !ok {
				return entries, fmt.Errorf("could not find corresponding game ID to %+v", entry[csvGameNameIndex])
			}
			parentID, ok := eventNameToIDMap[entry[csvParentEventIndex]]
			if !ok {
				return entries, fmt.Errorf("could not find corresponding parent event ID to %+v", entry[csvParentEventIndex])
			}
			imageID := entry[csvEventImageIndex]
			if !strings.HasPrefix(imageID, "gea:") {
				imageID, ok = imageURLToIDMap[entry[csvEventImageIndex]]
				if !ok {
					return entries, fmt.Errorf("could not find corresponding image ID to %+v", entry[csvEventImageIndex])
				}
			}

			event := Event{
				ID:          id,
				Title:       title,
				Description: description,
				StartAt:     startAt,
				EndAt:       endAt,
				ChannelID:   channelID,
				GameID:      gameID,
				ImageID:     imageID,
				ParentID:    parentID,
			}

			var newID string
			if geaHostTarget == utilityModeDryRun {
				newID = uuid.NewV4().String()
			} else {
				newID, err = generateEventLeaf(event)
				if err != nil {
					return entries, fmt.Errorf("could not generate event %+v=%+v", event.Title, err)
				}
			}
			if id == "" {
				entry[csvEventIDIndex] = newID
				entries[index] = entry
			}
			eventNameToIDMap[title] = entry[csvEventIDIndex]
		}
	}
	return entries, nil
}

func writeEntriesToCSV(entries [][]string, succeeded bool) {
	status := "SUCCESS"
	if !succeeded {
		status = "FAILED"
	}
	file, err := os.Create(outputDirectory + fmt.Sprint(time.Now().Unix()) + "-" + status + "-output.csv")
	if err != nil {
		log.Fatal("failed to create output file")
	}
	defer file.Close()
	csvWriter := csv.NewWriter(file)

	output := make([][]string, len(entries)+1)
	output[0] = csvColumnNames
	for i := 1; i < len(output); i++ {
		output[i] = entries[i-1]
	}

	err = csvWriter.WriteAll(output)
	if err != nil {
		log.Fatal("failed to write csv result to file")
	}
}

// Converts an Event struct into a gea TimetableEvent.
// Returns the gea eventID on success.
func generateEventCollection(event Event) (string, error) {
	imageID := &event.ImageID
	if *utilityMode != utilityModeProduction {
		imageID = nil
	}

	if event.ID != "" {
		english := "en"

		params := geaclient.UpdateTimetableEventParams{
			OwnerID:     &event.ChannelID,
			ImageID:     imageID,
			Language:    &english,
			Title:       &event.Title,
			Description: &event.Description,
			TimeZoneID:  &event.TimeZoneID,
		}
		geaEvent, err := eventsClient.UpdateTimetableEvent(context.Background(), event.ID, params, event.ChannelID, nil)
		if err != nil {
			return "", fmt.Errorf("error updating event collection %+v=%+v", event.Title, err)
		}
		return geaEvent.GetID(), nil
	}
	params := geaclient.CreateTimetableEventParams{
		OwnerID:     event.ChannelID,
		ImageID:     imageID,
		Language:    "en",
		Title:       event.Title,
		Description: event.Description,
		TimeZoneID:  event.TimeZoneID,
	}
	geaEvent, err := eventsClient.CreateTimetableEvent(context.Background(), params, event.ChannelID, nil)
	if err != nil {
		return "", fmt.Errorf("error creating event collection %+v=%+v", event.Title, err)
	}
	return geaEvent.GetID(), nil
}

// Converts an Event struct into a gea SegmentEvent.
// Returns the gea eventID on success.
func generateEventLeaf(event Event) (string, error) {
	imageID := &event.ImageID
	if *utilityMode != utilityModeProduction {
		imageID = nil
	}

	if event.ID != "" {
		params := geaclient.UpdateSegmentEventParams{
			OwnerID:     &event.ChannelID,
			ChannelID:   &event.ChannelID,
			StartTime:   &event.StartAt,
			EndTime:     &event.EndAt,
			ImageID:     imageID,
			Title:       &event.Title,
			Description: &event.Description,
			GameID:      &event.GameID,
		}
		geaEvent, err := eventsClient.UpdateSegmentEvent(context.Background(), event.ID, params, event.ChannelID, nil)
		if err != nil {
			return "", fmt.Errorf("error updating event leaf %+v=%+v", event.Title, err)
		}
		return geaEvent.GetID(), nil
	}
	params := geaclient.CreateSegmentEventParams{
		OwnerID:     event.ChannelID,
		ChannelID:   event.ChannelID,
		ParentID:    event.ParentID,
		StartTime:   event.StartAt,
		EndTime:     event.EndAt,
		ImageID:     imageID,
		Title:       event.Title,
		Description: event.Description,
		GameID:      event.GameID,
	}
	geaEvent, err := eventsClient.CreateSegmentEvent(context.Background(), params, event.ChannelID, nil)
	if err != nil {
		return "", fmt.Errorf("error creating event leaf %+v=%+v", event.Title, err)
	}
	return geaEvent.GetID(), nil
}

func convertToTimestamp(ts string, timezone string) (time.Time, error) {
	t, err := time.Parse(layout1, ts+" "+timezone)
	if err != nil {
		return time.Parse(layout2, ts+" "+timezone)
	}
	return t, err
}
