package clients

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"

	"strings"

	"code.justin.tv/feeds/errors"
	"golang.org/x/net/context"
)

const (
	maxBodySlurpSize = int64(2 << 10)
)

type errResponse struct {
	code int
	body bytes.Buffer
	id   string
}

func (e *errResponse) Error() string {
	return fmt.Sprintf("status_code=%d; id=%s; body=%s", e.code, e.id, e.body.String())
}

func (e *errResponse) HTTPCode() int {
	return e.code
}

func (e *errResponse) ID() string {
	return e.id
}

func (e *errResponse) Body() *bytes.Buffer {
	return &e.body
}

func (e *errResponse) IsThrottled() bool {
	return e.code == http.StatusTooManyRequests
}

func httpCleanup(body io.ReadCloser, maxBodySlurpSize int64, retErr *error) {
	_, err2 := io.CopyN(ioutil.Discard, body, maxBodySlurpSize)
	if err2 != io.EOF && err2 != nil && *retErr == nil {
		*retErr = errors.Wrap(err2, "unable to drain HTTP body")
	}
	err2 = body.Close()
	if err2 != nil && *retErr == nil {
		*retErr = errors.Wrap(err2, "unable to close HTTP body")
	}
}

// NewHTTPRequest accepts parameters for a request and returns an *http.Request if successful
type NewHTTPRequest func(context.Context, string, string, io.Reader) (*http.Request, error)

// HTTPNewRequest is the canonical implementation of NewHTTPRequest
func HTTPNewRequest(ctx context.Context, method string, url string, body io.Reader) (*http.Request, error) {
	req, err := http.NewRequest(method, url, body)
	if err != nil {
		return nil, err
	}
	req = req.WithContext(ctx)
	return req, nil
}

var _ NewHTTPRequest = HTTPNewRequest

// DoHTTP does common http stuff for getting a JSON response.  Using httpNewRequest so we can abstract out twitchhttp
// into clients that require it.
func do(ctx context.Context, method string, url string, queryParams url.Values, headers http.Header, body interface{}, into interface{}, httpNewRequest NewHTTPRequest, doer RequestDoer) (retErr error) {
	var bodyReader io.Reader
	if body != nil {
		buf := &bytes.Buffer{}
		if err := json.NewEncoder(buf).Encode(&body); err != nil {
			return errors.Wrap(err, "unable to encode message body")
		}
		bodyReader = buf
	}
	req, err := httpNewRequest(ctx, method, url, bodyReader)
	if err != nil {
		return errors.Wrap(err, "unable to make new HTTP request")
	}

	if len(queryParams) > 0 {
		req.URL.RawQuery = queryParams.Encode()
	}

	// copy the header map, to avoid any possible race conditions
	for k := range headers {
		req.Header.Set(k, headers.Get(k))
	}

	resp, err := doer.Do(req)
	if err != nil {
		return errors.Wrap(err, "unable to execute HTTP request")
	}
	defer httpCleanup(resp.Body, maxBodySlurpSize, &retErr)
	if err := handleRespForDo(resp); err != nil {
		return errors.Wrap(err, "unable to handle HTTP response")
	}
	if into != nil {
		if err := json.NewDecoder(resp.Body).Decode(into); err != nil {
			return errors.Wrap(err, "unable to decode HTTP body")
		}
	}
	return nil
}

func requestIDFromRes(resp *http.Response) string {
	traceID := resp.Header.Get("X-Amzn-Trace-Id")
	parts := strings.Split(traceID, ";")
	for _, part := range parts {
		section := strings.Split(part, "=")
		if len(section) != 2 {
			continue
		}
		if strings.ToLower(strings.TrimSpace(section[0])) == "root" {
			return strings.TrimSpace(section[1])
		}
	}
	return ""
}

func handleRespForDo(res *http.Response) error {
	if res.StatusCode != http.StatusOK {
		errVal := &errResponse{
			code: res.StatusCode,
			id:   requestIDFromRes(res),
		}
		_, err2 := io.CopyN(&errVal.body, res.Body, maxBodySlurpSize)
		if err2 != nil && err2 != io.EOF {
			return errors.Wrap(err2, "unable to copy HTTP body")
		}
		return errors.Wrap(errVal, "unexpected HTTP status code")
	}
	return nil
}

// RequestDoer is anything that can issue HTTP requests.  Usually a http.Client
type RequestDoer interface {
	Do(req *http.Request) (*http.Response, error)
}

// DoHTTP does common http stuff using an http client for getting a JSON response
func DoHTTP(ctx context.Context, client RequestDoer, method string, path string, queryParams url.Values, body interface{}, into interface{}, httpNewRequest NewHTTPRequest) error {
	return DoHTTPWithHeaders(ctx, client, method, path, queryParams, nil, body, into, httpNewRequest)
}

// DoHTTPWithHeaders does common http stuff using an http client for getting a JSON response.
// DoHTTPWithHeaders allows the caller specify the headers that are attached to a request.
func DoHTTPWithHeaders(ctx context.Context, client RequestDoer, method string, path string, queryParams url.Values, header http.Header, body interface{}, into interface{}, httpNewRequest NewHTTPRequest) error {
	if client == nil {
		client = http.DefaultClient
	}
	if httpNewRequest == nil {
		httpNewRequest = HTTPNewRequest
	}
	return do(ctx, method, path, queryParams, header, body, into, httpNewRequest, client)
}

// errorWithCode is an error type that specifies its HTTP status code.  Errors returned by feeds services typically
// implement this interface.
type errorWithCode interface {
	HTTPCode() int
}

// ErrorCode returns the HTTP status code that is part of the error, or 0 if there is no status code in the error.
func ErrorCode(err error) int {
	if coded, ok := errors.Cause(err).(errorWithCode); ok {
		return coded.HTTPCode()
	}
	return 0
}
