package twitchclient

import (
	"context"
	"fmt"
	"math/rand"
	"net"
	"regexp"
	"strings"
	"time"

	"github.com/cactus/go-statsd-client/statsd"
)

type ctxDialer interface {
	DialContext(ctx context.Context, network string, address string) (net.Conn, error)
}

type dialWrap struct {
	dialer   ctxDialer
	lookupIP func(host string) ([]net.IP, error)

	stats               statsd.Statter
	statPrefix          string
	enableExtraDNSStats bool
}

func dialerWithDNSStats(dialer ctxDialer, stats statsd.Statter, statPrefix string, enableExtraDNSStats bool) *dialWrap {
	return &dialWrap{
		dialer:   dialer,
		lookupIP: net.LookupIP,

		stats:               stats,
		statPrefix:          strings.TrimSuffix(statPrefix, "."), // make sure the prefix has no trailing "."
		enableExtraDNSStats: enableExtraDNSStats,
	}
}

func (d *dialWrap) DialContext(ctx context.Context, network, address string) (net.Conn, error) {
	if d.stats == nil {
		return d.dialer.DialContext(ctx, network, address) // If we aren't tracking DNS stats, simply use the default dial
	}
	start := time.Now()

	// TCP connections should always have host and port
	host, port, err := net.SplitHostPort(address)
	if err != nil {
		return nil, err
	}

	// If host is an IP, no DNS lookup is needed
	// NOTE: net.(r *Resolver).LookupIPAddr is already doing this (may be ok to remove this lines)
	if net.ParseIP(host) != nil {
		return d.dialer.DialContext(ctx, network, address)
	}

	// DNS lookup
	ips, err := d.lookupIP(host)
	if err != nil {
		_ = d.stats.Inc(d.statFailure(host), 1, 1.0)
		return nil, err
	}

	ips = filterValidIPs(network, ips)
	if len(ips) == 0 {
		_ = d.stats.Inc(d.statFailure(host), 1, 1.0)
		return nil, fmt.Errorf("twitchclient: dns lookup had no results for network: %q, host: %q", host, network)
	}

	ip := ips[rand.Intn(len(ips))] // pick random IP

	// Stats
	duration := time.Since(start)
	_ = d.stats.TimingDuration(d.statSuccess(host), duration, 1.0)
	if d.enableExtraDNSStats {
		_ = d.stats.Inc(d.statIP(host, ip.String()), 1, 1.0)
		_ = d.stats.Gauge(d.statUniqIPs(host), int64(len(ips)), 1.0)
	}

	return d.dialer.DialContext(ctx, network, net.JoinHostPort(ip.String(), port))
}

// filter out any IPs that aren't the correct version, only TCP is allowed
func filterValidIPs(network string, ips []net.IP) []net.IP {
	var validIPs []net.IP
	for _, ip := range ips {
		if (network == "tcp4" && ip.To4() == nil) || (network == "tcp6" && ip.To16() == nil) {
			continue
		}
		validIPs = append(validIPs, ip)
	}
	return validIPs
}

func (d *dialWrap) statSuccess(host string) string {
	return fmt.Sprintf("%s.%s.success", d.statPrefix, sanitizeStat(host))
}

func (d *dialWrap) statFailure(host string) string {
	return fmt.Sprintf("%s.%s.failure", d.statPrefix, sanitizeStat(host))
}

func (d *dialWrap) statIP(host, ip string) string {
	return fmt.Sprintf("%s.%s.%s", d.statPrefix, sanitizeStat(host), sanitizeStat(ip))
}

func (d *dialWrap) statUniqIPs(host string) string {
	return fmt.Sprintf("%s.%s.%s", d.statPrefix, sanitizeStat(host), "uniq_ips")
}

var invalidStatChars = regexp.MustCompile(`[^A-Za-z0-9-]+`)

// sanitizestat replaces all invalid characters with "_"
func sanitizeStat(stat string) string {
	return invalidStatChars.ReplaceAllString(stat, "_")
}
