package twitchclient

import (
	"bytes"
	"errors"
	"io"
	"io/ioutil"
	"net/http"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

func TestHandleFailedResponse(t *testing.T) {
	Convey("Handles Failed Responses", t, func() {
		resp := &http.Response{StatusCode: http.StatusInternalServerError, Header: http.Header{}}

		Convey("Response is json", func() {
			resp.Header.Add("Content-Type", "application/json")
			resp.Body = ioutil.NopCloser(bytes.NewBufferString(`{ "Message": "Hello" }`))
			err := *HandleFailedResponse(resp)
			So(err.StatusCode, ShouldEqual, http.StatusInternalServerError)
			So(err.Message, ShouldEqual, "Hello")
		})
		Convey("Response is plain text", func() {
			resp.Body = ioutil.NopCloser(bytes.NewBufferString("Imagine this is HTML"))

			err := *HandleFailedResponse(resp)
			So(err.StatusCode, ShouldEqual, http.StatusInternalServerError)
			So(err.Message, ShouldEqual, "Imagine this is HTML")
		})
		Convey("Response is unreadable plain text", func() {
			resp.Body = &readErrorReadCloser{}
			err := *HandleFailedResponse(resp)
			So(err.StatusCode, ShouldEqual, http.StatusInternalServerError)
			So(err.Message, ShouldEqual, "Unable to read response body: unexpected EOF")
		})
	})
}

type closeErrorReadCloser struct {
	io.Reader
}

type readErrorReadCloser struct{}

func (e *readErrorReadCloser) Read(buffer []byte) (int, error) {
	return 0, errors.New("unexpected EOF")
}

func (e *readErrorReadCloser) Close() error {
	return nil
}

func (e *closeErrorReadCloser) Close() error {
	return errors.New("could not close")
}
