package xray

import (
	"context"
	"encoding/json"
	"net"
	"net/http"
	"os"
	"strconv"
	"time"
)

var (
	listenerAddr = &net.UDPAddr{
		IP:   net.IPv4(127, 0, 0, 1),
		Port: 2000,
	}

	testDaemon = &testdaemon{
		Channel: make(chan *result, 200),
	}
)

func init() {
	if testDaemon.Connection == nil {
		conn, err := net.ListenUDP("udp", listenerAddr)
		if err != nil {
			panic(err)
		}

		testDaemon.Connection = conn
		go testDaemon.Run()
	}
}

type testdaemon struct {
	Connection *net.UDPConn
	Channel    chan *result
	Done       bool
}
type result struct {
	Segment *segment
	Error   error
}

func (td *testdaemon) Run() {
	buffer := make([]byte, maxSegmentSize)
	for !td.Done {
		n, _, err := td.Connection.ReadFromUDP(buffer)
		if err != nil {
			td.Channel <- &result{nil, err}
			continue
		}

		buffered := buffer[len(header):n]

		seg := &segment{}
		err = json.Unmarshal(buffered, &seg)
		if err != nil {
			td.Channel <- &result{nil, err}
			continue
		}

		seg.sampled = true
		td.Channel <- &result{seg, err}
	}
}

func (td *testdaemon) Recv() (*segment, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 100*time.Millisecond)
	defer cancel()
	select {
	case r := <-td.Channel:
		return r.Segment, r.Error
	case <-ctx.Done():
		return nil, ctx.Err()
	}
}

type xRayHeaders struct {
	RootTraceID string
	ParentID    string
	Sampled     bool
}

func parseHeadersForTest(h http.Header) xRayHeaders {
	m := parseHeaders(h)
	s, _ := strconv.ParseBool(m["Sampled"])

	return xRayHeaders{
		RootTraceID: m["Root"],
		ParentID:    m["Parent"],
		Sampled:     s,
	}
}

func resetConfig() {
	config = &privateConfig{
		name:        os.Getenv("XRAY_TRACING_NAME"),
		defaultName: os.Getenv("XRAY_TRACING_DEFAULT_NAME"),
		daemonAddr: &net.UDPAddr{
			IP:   net.IPv4(127, 0, 0, 1),
			Port: 2000,
		},
		sampling: 1.0,
	}
}
