package geaclient

import (
	"context"
	"fmt"
	"net/url"
	"strings"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/clients/twitchdoer"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/foundation/twitchclient"
	"github.com/google/go-querystring/query"
)

const (
	defaultTimingXactName = "events-gea"
	defaultStatPrefix     = "service.events_gea."
	defaultStatSampleRate = 1.0

	EventTypeSingle    = "single"
	EventTypePremiere  = "premiere"
	EventTypeTimetable = "timetable"
	EventTypeSegment   = "segment"

	EventSortByHype      = "hype"
	EventSortByStartTime = "start_time"
)

// Client satisfies visage's interface requirements for clients
type Client interface {
	CreateSingleEvent(ctx context.Context, params CreateSingleEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*SingleEvent, error)
	UpdateSingleEvent(ctx context.Context, eventID string, params UpdateSingleEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*SingleEvent, error)
	// Deprecated: use DeleteEvent instead
	DeleteSingleEvent(ctx context.Context, eventID string, userID string, reqOpts *twitchclient.ReqOpts) (*SingleEvent, error)

	CreatePremiereEvent(ctx context.Context, params CreatePremiereEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*PremiereEvent, error)
	UpdatePremiereEvent(ctx context.Context, eventID string, params UpdatePremiereEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*PremiereEvent, error)
	// Deprecated: use DeleteEvent instead
	DeletePremiereEvent(ctx context.Context, eventID string, userID string, reqOpts *twitchclient.ReqOpts) (*PremiereEvent, error)

	CreateTimetableEvent(ctx context.Context, params CreateTimetableEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*TimetableEvent, error)
	UpdateTimetableEvent(ctx context.Context, eventID string, params UpdateTimetableEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*TimetableEvent, error)

	CreateSegmentEvent(ctx context.Context, params CreateSegmentEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*SegmentEvent, error)
	UpdateSegmentEvent(ctx context.Context, eventID string, params UpdateSegmentEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*SegmentEvent, error)

	DeleteEvent(ctx context.Context, eventID string, userID string, reqOpts *twitchclient.ReqOpts) (Event, error)

	GetEvent(ctx context.Context, eventID string, options *GetEventOpts, reqOpts *twitchclient.ReqOpts) (Event, error)
	GetEvents(ctx context.Context, eventIDs []string, options *GetEventsOpts, reqOpts *twitchclient.ReqOpts) ([]Event, error)
	GetLiveEvent(ctx context.Context, channelID string, options *GetEventsOpts, reqOpts *twitchclient.ReqOpts) (Event, error)
	GetEventMetadata(ctx context.Context, eventID string, options *GetEventMetadataOpts, reqOpts *twitchclient.ReqOpts) (*EventMetadata, error)
	GetEventStats(ctx context.Context, eventIDs []string, reqOpts *twitchclient.ReqOpts) (*EventStatsResponse, error)

	GetEventVideos(ctx context.Context, eventID string, options *GetEventArchiveVideosOpts, reqOpts *twitchclient.ReqOpts) (*EventVideos, error)

	GetEventIDsByChannelIDs(ctx context.Context, channelIDs []string, options *GetEIDsByCIDsOpts, reqOpts *twitchclient.ReqOpts) (*EventIDs, error)
	GetEventIDsByTypes(ctx context.Context, types []string, options *GetEIDsByTypesOpts, reqOpts *twitchclient.ReqOpts) (*EventIDs, error)
	GetEventIDsByFilterOptions(ctx context.Context, options *GetEIDsFilterOpts, reqOpts *twitchclient.ReqOpts) (*EventIDsFromFilterOpts, error)

	GetEventSuggestionsForGame(ctx context.Context, gameID string, options *GetSuggestionsOpts, reqOpts *twitchclient.ReqOpts) (*SuggestedEventIDs, error)

	AddLocalization(ctx context.Context, params AddLocalizationParams, userID string, reqOpts *twitchclient.ReqOpts) (*Localization, error)
	UpdateLocalization(ctx context.Context, params UpdateLocalizationParams, userID string, reqOpts *twitchclient.ReqOpts) (*Localization, error)
	RemoveLocalization(ctx context.Context, eventID string, language string, userID string, reqOpts *twitchclient.ReqOpts) (*Localization, error)

	FollowEvent(ctx context.Context, eventID string, userID string, reqOpts *twitchclient.ReqOpts) error
	UnfollowEvent(ctx context.Context, eventID string, userID string, reqOpts *twitchclient.ReqOpts) error
	GetEventFollowers(ctx context.Context, eventID string, options *GetEventFollowersOptions, reqOpts *twitchclient.ReqOpts) (*EventFollowers, error)
	GetFollowedEvents(ctx context.Context, userID string, options *GetFollowedEventsOptions, reqOpts *twitchclient.ReqOpts) (*FollowedEvents, error)
	CheckFollowedEvents(ctx context.Context, eventIDs []string, userID string, reqOpts *twitchclient.ReqOpts) (*FollowedEvents, error)

	ReserveImageUpload(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*ImageUploadReservation, error)

	SendEventStartedNotifications(ctx context.Context, eventIDs []string, reqOpts *twitchclient.ReqOpts) error

	// DEPRECATED API, PLEASE USE ReserveImageUpload AND UPLOAD TO GENERIC UPLOAD DIRECT
	UploadBase64Image(ctx context.Context, userID string, imagePayload *ImagePayload, reqOpts *twitchclient.ReqOpts) (*ImageUploadConfirmation, error)
}

// ClientImpl implements the Client interface and uses the twitchhttp client to make http requests
type ClientImpl struct {
	Client twitchclient.Client
}

var _ Client = &ClientImpl{}

// NewClient returns a new instance of the Client which uses the given config
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}
	return &ClientImpl{Client: twitchClient}, nil
}

func (c *ClientImpl) http(ctx context.Context, statName string, method string, path string, queryParams url.Values, body interface{}, reqOpts *twitchclient.ReqOpts, into interface{}) error {
	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       defaultStatPrefix + statName,
		StatSampleRate: defaultStatSampleRate,
	})

	doer := &twitchdoer.TwitchHTTPDoer{
		Client: c.Client,
		Reqopt: combinedReqOpts,
	}
	return clients.DoHTTP(ctx, doer, method, path, queryParams, body, into, doer.NewTwitchRequest)
}

func validateNotNil(fields map[string]interface{}) error {
	for k, v := range fields {
		if v == nil {
			return errors.Errorf("event field %s was nil but it shouldn't be", k)
		}
	}
	return nil
}

type createEventParams struct {
	OwnerID  string  `json:"owner_id"`
	Type     string  `json:"type"`
	ParentID *string `json:"parent_id"`

	StartTime *time.Time `json:"start_time"`
	EndTime   *time.Time `json:"end_time"`
	// TimeZoneID is expected to be a tz database time zone name.  E.g. America/New_York
	TimeZoneID *string `json:"time_zone_id"`

	ImageID     *string `json:"image_id"`
	Language    *string `json:"language"`
	Title       *string `json:"title"`
	Description *string `json:"description"`
	ChannelID   *string `json:"channel_id"`
	GameID      *string `json:"game_id"`

	// Premiere fields
	PremiereID *string `json:"premiere_id"`
}

func (c *ClientImpl) createEvent(ctx context.Context, statName string, params createEventParams, into interface{}, userID string, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/v1/create_event?user_id=%s", url.QueryEscape(userID))

	fmt.Println(path)

	return c.http(ctx, statName, "POST", path, nil, params, reqOpts, into)
}

type updateEventParams struct {
	ID       string  `json:"id"`
	OwnerID  *string `json:"owner_id"`
	Type     *string `json:"type"`
	ParentID *string `json:"parent_id"`

	StartTime *time.Time `json:"start_time"`
	EndTime   *time.Time `json:"end_time"`
	// TimeZoneID is expected to be a tz database time zone name.  E.g. America/New_York
	TimeZoneID *string `json:"time_zone_id"`

	ImageID     *string `json:"image_id"`
	Language    *string `json:"language"`
	Title       *string `json:"title"`
	Description *string `json:"description"`
	ChannelID   *string `json:"channel_id"`
	GameID      *string `json:"game_id"`

	// Premiere fields
	PremiereID *string `json:"premiere_id"`
}

func (c *ClientImpl) updateEvent(ctx context.Context, statName string, params updateEventParams, into interface{}, userID string, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/v1/update_event?user_id=%s", url.QueryEscape(userID))

	return c.http(ctx, statName, "POST", path, nil, params, reqOpts, into)
}

type deleteEventParams struct {
	ID string `json:"id"`
}

func (c *ClientImpl) DeleteEvent(ctx context.Context, eventID string, userID string, reqOpts *twitchclient.ReqOpts) (Event, error) {
	path := fmt.Sprintf("/v1/delete_event?user_id=%s", url.QueryEscape(userID))
	params := &deleteEventParams{ID: eventID}

	event := &eventSuperstruct{}
	err := c.http(ctx, "delete_event", "POST", path, nil, params, reqOpts, event)
	if err != nil {
		return nil, err
	}
	return fromEventSuperstruct(event)
}

type Event interface {
	GetID() string
	GetType() string
}

// Only used for decoding, so no need for omitempty
type eventSuperstruct struct {
	ID       string  `json:"id"`
	OwnerID  string  `json:"owner_id"`
	Type     string  `json:"type"`
	ParentID *string `json:"parent_id"`

	CreatedAt time.Time  `json:"created_at"`
	UpdatedAt time.Time  `json:"updated_at"`
	DeletedAt *time.Time `json:"deleted_at"`

	StartTime  *time.Time `json:"start_time"`
	EndTime    *time.Time `json:"end_time"`
	TimeZoneID *string    `json:"time_zone_id"`

	ImageID     *string `json:"image_id"`
	ImageURL    *string `json:"image_url"`
	Language    *string `json:"language"`
	Title       *string `json:"title"`
	Description *string `json:"description"`
	ChannelID   *string `json:"channel_id"`
	GameID      *string `json:"game_id"`

	// Premiere fields
	PremiereID *string `json:"premiere_id"`

	// Timetable fields
	ChannelIDs []string `json:"channel_ids"`
	GameIDs    []string `json:"game_ids"`
}

type UnknownEvent struct {
	ID   string
	Type string
}

func (c UnknownEvent) GetID() string {
	return c.ID
}

func (c UnknownEvent) GetType() string {
	return c.Type
}

func ErrorCode(err error) int {
	return clients.ErrorCode(err)
}

type GetEventOpts struct {
	GetDeleted bool  `url:"deleted"`
	SkipCache  *bool `url:"skip_cache"`
}

func (c *ClientImpl) GetEvent(ctx context.Context, eventID string, options *GetEventOpts, reqOpts *twitchclient.ReqOpts) (Event, error) {
	path := "/v1/get_event"

	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("id", eventID)

	event := &eventSuperstruct{}
	err = c.http(ctx, "get_event", "GET", path, query, nil, reqOpts, event)
	if err != nil {
		return nil, err
	}
	return fromEventSuperstruct(event)
}

func fromEventSuperstruct(superstruct *eventSuperstruct) (Event, error) {
	switch superstruct.Type {
	case EventTypeSingle:
		event := &SingleEvent{}
		err := event.fromSuperstruct(superstruct)
		if err != nil {
			return nil, err
		}
		return event, nil
	case EventTypePremiere:
		event := &PremiereEvent{}
		err := event.fromSuperstruct(superstruct)
		if err != nil {
			return nil, err
		}
		return event, nil
	case EventTypeTimetable:
		event := &TimetableEvent{}
		err := event.fromSuperstruct(superstruct)
		if err != nil {
			return nil, err
		}
		return event, nil
	case EventTypeSegment:
		event := &SegmentEvent{}
		err := event.fromSuperstruct(superstruct)
		if err != nil {
			return nil, err
		}
		return event, nil
	default:
		return &UnknownEvent{ID: superstruct.ID, Type: superstruct.Type}, nil
	}
}

type GetEventsOpts struct {
	GetDeleted bool  `url:"deleted"`
	SkipCache  *bool `url:"skip_cache"`
}

func (c *ClientImpl) GetEvents(ctx context.Context, eventIDs []string, options *GetEventsOpts, reqOpts *twitchclient.ReqOpts) ([]Event, error) {
	path := "/v1/get_events"

	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("ids", strings.Join(eventIDs, ","))

	var events []*eventSuperstruct
	err = c.http(ctx, "get_events", "GET", path, query, nil, reqOpts, &events)
	if err != nil {
		return nil, err
	}

	result := make([]Event, len(events))

	for i, event := range events {
		if event == nil {
			continue
		}
		ev, err := fromEventSuperstruct(event)
		if err != nil {
			return nil, err
		}
		result[i] = ev
	}

	return result, nil
}

func (c *ClientImpl) GetLiveEvent(ctx context.Context, channelID string, options *GetEventsOpts, reqOpts *twitchclient.ReqOpts) (Event, error) {
	path := "/v1/get_live_event"

	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("channel_id", channelID)

	var event *eventSuperstruct
	err = c.http(ctx, "get_live_event", "GET", path, query, nil, reqOpts, &event)
	if err != nil {
		return nil, err
	} else if event == nil {
		return nil, nil
	}
	return fromEventSuperstruct(event)
}

type EventMetadata struct {
	ID          string `json:"id"`
	Title       string `json:"title"`
	Description string `json:"description"`

	// Open Graph properties
	OGTitle       string `json:"og_title"`
	OGDescription string `json:"og_description"`
	OGURL         string `json:"og_url"`
	OGImage       string `json:"og_image"`
	OGImageWidth  string `json:"og_image_width"`
	OGImageHeight string `json:"og_image_height"`
	OGType        string `json:"og_type"`

	TwitterCard        string `json:"twitter_card"`
	TwitterTitle       string `json:"twitter_title"`
	TwitterDescription string `json:"twitter_description"`
	TwitterImage       string `json:"twitter_image"`
	TwitterURL         string `json:"twitter_url"`
}

type GetEventMetadataOpts struct {
	SkipCache *bool `url:"skip_cache"`
}

func (c *ClientImpl) GetEventMetadata(ctx context.Context, eventID string, options *GetEventMetadataOpts, reqOpts *twitchclient.ReqOpts) (*EventMetadata, error) {
	path := "/v1/get_event_metadata"

	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("id", eventID)

	event := &EventMetadata{}
	err = c.http(ctx, "get_event_metadata", "GET", path, query, nil, reqOpts, event)
	if err != nil {
		return nil, err
	}

	return event, err
}

type EventStats struct {
	EventID     string `json:"event_id"`
	FollowCount int    `json:"follow_count"`
}

type EventStatsResponse struct {
	Items []EventStats `json:"items"`
}

func (c *ClientImpl) GetEventStats(ctx context.Context, eventIDs []string, reqOpts *twitchclient.ReqOpts) (*EventStatsResponse, error) {
	path := fmt.Sprintf("/v1/get_event_stats?event_ids=%s", strings.Join(eventIDs, ","))

	var response EventStatsResponse
	err := c.http(ctx, "get_event_stats", "GET", path, nil, nil, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

type EventIDs struct {
	EventIDs []string `json:"event_ids"`
	Cursor   string   `json:"cursor"`
}

type EventIDsFromFilterOpts struct {
	EventIDs      []string `json:"event_ids"`
	LastIDCursor  string   `json:"cursor"`
	FirstIDCursor string   `json:"first_item_cursor,omitempty"`

	HasNextPage     bool `json:"has_next_page,omitempty"`
	HasPreviousPage bool `json:"has_previous_page,omitempty"`
}

type GetEIDsByCIDsOpts struct {
	// Include events that have a start time greater than or equal to the given parameter
	StartTimeAfter *time.Time `url:"start_time_after"`

	// Include events that have a start time less than the given parameter
	StartTimeBefore *time.Time `url:"start_time_before"`

	// Include events that have an end time greater than or equal to the given parameter
	EndTimeAfter *time.Time `url:"end_time_after"`

	// Include events that have an end time less than the given parameter
	EndTimeBefore *time.Time `url:"end_time_before"`

	Limit      *int    `url:"limit"`
	Cursor     *string `url:"cursor"`
	Descending *bool   `url:"desc"`

	// Bypasses the cache
	NoCache *bool `url:"no_cache"`
}

func (c *ClientImpl) GetEventIDsByChannelIDs(ctx context.Context, channelIDs []string, options *GetEIDsByCIDsOpts, reqOpts *twitchclient.ReqOpts) (*EventIDs, error) {
	path := "/v1/get_event_ids_by_channel_ids"
	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("channel_ids", strings.Join(channelIDs, ","))
	if options.EndTimeBefore != nil {
		query.Add("end_time_before", options.EndTimeBefore.Format(time.RFC3339))
	}
	if options.EndTimeAfter != nil {
		query.Add("end_time_after", options.EndTimeAfter.Format(time.RFC3339))
	}
	if options.StartTimeBefore != nil {
		query.Add("start_time_before", options.StartTimeBefore.Format(time.RFC3339))
	}
	if options.StartTimeAfter != nil {
		query.Add("start_time_after", options.StartTimeAfter.Format(time.RFC3339))
	}

	var response EventIDs
	err = c.http(ctx, "get_event_ids_by_channel_ids", "GET", path, query, nil, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

type GetEIDsByTypesOpts struct {
	// Include events that have a start time greater than or equal to the given parameter
	StartTimeAfter *time.Time `url:"start_time_after"`

	// Include events that have a start time less than the given parameter
	StartTimeBefore *time.Time `url:"start_time_before"`

	// Include events that have an end time greater than or equal to the given parameter
	EndTimeAfter *time.Time `url:"end_time_after"`

	// Include events that have an end time less than the given parameter
	EndTimeBefore *time.Time `url:"end_time_before"`

	Limit  *int    `url:"limit"`
	Cursor *string `url:"cursor"`
}

func (c *ClientImpl) GetEventIDsByTypes(ctx context.Context, types []string, options *GetEIDsByTypesOpts, reqOpts *twitchclient.ReqOpts) (*EventIDs, error) {
	path := "/v1/get_event_ids_by_types"
	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("types", strings.Join(types, ","))

	var response EventIDs
	err = c.http(ctx, "get_event_ids_by_types", "GET", path, query, nil, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

type GetEIDsFilterOpts struct {
	// Include events that are (immediate) children of the given parent event IDs
	ParentEventIDs []string `url:"parents,comma"`

	// Types specifies the types of events that should be returned.
	// E.g. "single", "premiere", "timetable"
	Types []string `url:"types,comma"`

	// Include events that involve playing one of the given games
	GameIDs []string `url:"game_ids,comma"`

	// Include events that are broadcast on one of the given channels
	ChannelIDs []string `url:"channel_ids,comma"`

	// Include events that are owned by a particular user
	OwnerIDs []string `url:"owner_ids,comma"`

	// Include events that have a start time greater than or equal to the given parameter
	StartTimeAfter *time.Time `url:"start_time_after"`

	// Include events that have a start time less than the given parameter
	StartTimeBefore *time.Time `url:"start_time_before"`

	// Include events that have an end time greater than or equal to the given parameter
	EndTimeAfter *time.Time `url:"end_time_after"`

	// Include events that have an end time less than the given parameter
	EndTimeBefore *time.Time `url:"end_time_before"`

	Limit *int `url:"limit"`

	Cursor *string `url:"cursor"`

	SortBy string `url:"sort_by"`

	Descending bool `url:"desc"`

	// Set GetPreviousPage to True to fetch the previous page of IDs, instead of the next page.
	GetPreviousPage bool `url:"get_prev_page"`

	FirstPageOptions []string `url:"first_page_options,comma"`

	// Bypass the cache, and do not cache results
	NoCache bool `url:"no_cache"`
}

func (o *GetEIDsFilterOpts) AddIDFirstPageOption(eventID string) {
	option := fmt.Sprintf("id:%s", eventID)
	o.FirstPageOptions = append(o.FirstPageOptions, option)
}

// Note that AddEndsAfterFirstPageOption rounds the given time to the latest minute to make the
// request more cacheable.
func (o *GetEIDsFilterOpts) AddEndsAfterFirstPageOption(endsAfter time.Time) {
	endsAfter = convertToCachableTime(endsAfter)
	option := fmt.Sprintf("ends_after:%s", endsAfter.Format(time.RFC3339))
	o.FirstPageOptions = append(o.FirstPageOptions, option)
}

// GetEventIDsByFilterOptions returns the event IDs specified by the given options.
// Note that GetEventIDsByFilterOptions rounds all given time parameters to the nearest minute to make the
// request more cacheable.
func (c *ClientImpl) GetEventIDsByFilterOptions(ctx context.Context, options *GetEIDsFilterOpts, reqOpts *twitchclient.ReqOpts) (*EventIDsFromFilterOpts, error) {
	path := "/v1/get_event_ids_by_filter"

	options = sanitizeGetEIDsFilterOpts(options)
	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}

	var response EventIDsFromFilterOpts
	err = c.http(ctx, "get_event_ids_by_filter", "GET", path, query, nil, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

func sanitizeGetEIDsFilterOpts(options *GetEIDsFilterOpts) *GetEIDsFilterOpts {
	if options == nil {
		return nil
	}

	updated := *options
	updated.EndTimeBefore = convertToCachableTimePtr(updated.EndTimeBefore)
	updated.EndTimeAfter = convertToCachableTimePtr(updated.EndTimeAfter)
	updated.StartTimeAfter = convertToCachableTimePtr(updated.StartTimeAfter)
	updated.StartTimeBefore = convertToCachableTimePtr(updated.StartTimeBefore)

	return &updated
}

func convertToCachableTimePtr(t *time.Time) *time.Time {
	if t == nil {
		return nil
	}
	cacheableTime := convertToCachableTime(*t)
	return &cacheableTime
}

// When the current time is used as part of a request, it can drastically lower cache hit rates.
// To make requests that this client sends more cacheable, convertToCachableTime rounds times to the
// nearest minute.
func convertToCachableTime(t time.Time) time.Time {
	t = t.UTC()
	return t.Round(time.Minute)
}

type SuggestedEventIDs struct {
	Live   []string `json:"live_ids"`
	Future []string `json:"future_ids"`
	Past   []string `json:"past_ids"`
}

type GetSuggestionsOpts struct {
	Languages []string `url:"languages"`
}

func (c *ClientImpl) GetEventSuggestionsForGame(ctx context.Context, gameID string, options *GetSuggestionsOpts, reqOpts *twitchclient.ReqOpts) (*SuggestedEventIDs, error) {
	path := "/v1/get_event_suggestions_by_game"

	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("game_id", gameID)

	var response SuggestedEventIDs
	err = c.http(ctx, "get_event_suggestions_by_game", "GET", path, query, nil, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

func (c *ClientImpl) FollowEvent(ctx context.Context, eventID string, userID string, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/v1/follow_event?event_id=%s&user_id=%s", url.QueryEscape(eventID), url.QueryEscape(userID))

	return c.http(ctx, "follow_event", "POST", path, nil, nil, reqOpts, nil)
}

func (c *ClientImpl) UnfollowEvent(ctx context.Context, eventID string, userID string, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/v1/unfollow_event?event_id=%s&user_id=%s", url.QueryEscape(eventID), url.QueryEscape(userID))

	return c.http(ctx, "unfollow_event", "POST", path, nil, nil, reqOpts, nil)
}

type GetEventFollowersOptions struct {
	Limit  *int    `url:"limit"`
	Cursor *string `url:"cursor"`
}

type EventFollowers struct {
	UserIDs []string `json:"user_ids"`
	Cursor  string   `json:"cursor"`
}

func (c *ClientImpl) GetEventFollowers(ctx context.Context, eventID string, options *GetEventFollowersOptions, reqOpts *twitchclient.ReqOpts) (*EventFollowers, error) {
	path := "/v1/get_event_followers"
	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("event_id", eventID)

	var response EventFollowers
	err = c.http(ctx, "get_event_followers", "GET", path, query, nil, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

type FollowedEvents struct {
	EventIDs []string `json:"event_ids"`
	Cursor   string   `json:"cursor"`
}

type GetFollowedEventsOptions struct {
	Limit  *int    `url:"limit"`
	Cursor *string `url:"cursor"`
}

func (c *ClientImpl) GetFollowedEvents(ctx context.Context, userID string, options *GetFollowedEventsOptions, reqOpts *twitchclient.ReqOpts) (*FollowedEvents, error) {
	path := "/v1/get_followed_events"
	values, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	values.Add("user_id", userID)

	var response FollowedEvents
	err = c.http(ctx, "get_followed_events", "GET", path, values, nil, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

func (c *ClientImpl) CheckFollowedEvents(ctx context.Context, eventIDs []string, userID string, reqOpts *twitchclient.ReqOpts) (*FollowedEvents, error) {
	path := fmt.Sprintf("/v1/check_followed_events?event_ids=%s&user_id=%s", strings.Join(eventIDs, ","), url.QueryEscape(userID))

	var response FollowedEvents
	err := c.http(ctx, "check_followed_events", "GET", path, nil, nil, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

func (c *ClientImpl) SendEventStartedNotifications(ctx context.Context, eventIDs []string, reqOpts *twitchclient.ReqOpts) error {
	path := "/v1/send_event_started_notifications"

	query := url.Values{}
	query.Add("ids", strings.Join(eventIDs, ","))

	return c.http(ctx, "send_event_started_notifications", "POST", path, query, nil, reqOpts, nil)
}
