package geaclient

import (
	"context"
	"time"

	"code.justin.tv/foundation/twitchclient"
)

type TimetableEvent struct {
	ID      string `json:"id"`
	OwnerID string `json:"owner_id"`
	Type    string `json:"type"`

	CreatedAt time.Time  `json:"created_at"`
	UpdatedAt time.Time  `json:"updated_at"`
	DeletedAt *time.Time `json:"deleted_at,omitempty"`

	StartTime *time.Time `json:"start_time,omitempty"`
	EndTime   *time.Time `json:"end_time,omitempty"`
	// TimeZoneID is a tz database time zone name.  E.g. America/New_York
	TimeZoneID string `json:"time_zone_id"`

	ImageID     *string  `json:"image_id,omitempty"`
	ImageURL    string   `json:"image_url"`
	Language    string   `json:"language"`
	Title       string   `json:"title"`
	Description string   `json:"description"`
	ChannelIDs  []string `json:"channel_ids"`
	GameIDs     []string `json:"game_ids"`
}

func (c TimetableEvent) GetID() string {
	return c.ID
}

func (c TimetableEvent) GetType() string {
	return c.Type
}

func (e *TimetableEvent) fromSuperstruct(s *eventSuperstruct) error {
	err := validateNotNil(map[string]interface{}{
		"time_zone_id": s.TimeZoneID,
		"image_url":    s.ImageURL,
		"language":     s.Language,
		"title":        s.Title,
		"description":  s.Description,
	})
	if err != nil {
		return err
	}
	e.ID = s.ID
	e.OwnerID = s.OwnerID
	e.Type = EventTypeTimetable

	e.CreatedAt = s.CreatedAt
	e.UpdatedAt = s.UpdatedAt
	e.DeletedAt = s.DeletedAt

	e.StartTime = s.StartTime
	e.EndTime = s.EndTime
	e.TimeZoneID = *s.TimeZoneID

	e.ImageID = s.ImageID
	e.ImageURL = *s.ImageURL
	e.Language = *s.Language
	e.Title = *s.Title
	e.Description = *s.Description
	e.ChannelIDs = s.ChannelIDs
	e.GameIDs = s.GameIDs

	return nil
}

type CreateTimetableEventParams struct {
	OwnerID string

	// TimeZoneID is expected to be a tz database time zone name.  E.g. America/New_York
	TimeZoneID string

	ImageID     *string
	Language    string
	Title       string
	Description string
}

func (c *ClientImpl) CreateTimetableEvent(ctx context.Context, params CreateTimetableEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*TimetableEvent, error) {
	var response TimetableEvent
	err := c.createEvent(ctx, "create_timetable_event", createEventParams{
		OwnerID:     params.OwnerID,
		Type:        EventTypeTimetable,
		TimeZoneID:  &params.TimeZoneID,
		ImageID:     params.ImageID,
		Language:    &params.Language,
		Title:       &params.Title,
		Description: &params.Description,
	}, &response, userID, reqOpts)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

type UpdateTimetableEventParams struct {
	OwnerID *string

	// TimeZoneID is expected to be a tz database time zone name.  E.g. America/New_York
	TimeZoneID *string

	ImageID     *string
	Language    *string
	Title       *string
	Description *string
}

func (c *ClientImpl) UpdateTimetableEvent(ctx context.Context, eventID string, params UpdateTimetableEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*TimetableEvent, error) {
	var response TimetableEvent
	err := c.updateEvent(ctx, "update_timetable_event", updateEventParams{
		ID:          eventID,
		OwnerID:     params.OwnerID,
		TimeZoneID:  params.TimeZoneID,
		ImageID:     params.ImageID,
		Language:    params.Language,
		Title:       params.Title,
		Description: params.Description,
	}, &response, userID, reqOpts)
	if err != nil {
		return nil, err
	}
	return &response, nil
}
