package geaclient

import (
	"context"
	"time"

	"code.justin.tv/foundation/twitchclient"
	"github.com/google/go-querystring/query"
)

const (
	VideoSortByStartTime = "start_time"
	VideoSortByViews     = "views"
)

const (
	VideoTypeArchive   = "archive"
	VideoTypeHighlight = "highlight"
)

type GetEventArchiveVideosOpts struct {
	Limit       *int    `url:"limit"`
	AfterCursor *string `url:"after_cursor"`

	// SortBy should be one of the constants, VideoSortByStartTime or VideoSortByViews.
	// If left unspecified, defaults to VideoSortByStartTime.
	// Note that videos will be sorted in descending order.
	SortBy *string `url:"sort_by"`

	// VideoType should be one of the constants, VideoTypeArchive or VideoTypeHighlight
	VideoType *string `url:"video_type"`
}

// EventVideos is a page of a list of event videos.
type EventVideos struct {
	Items       []EventVideo `json:"items"`
	HasNextPage bool         `json:"has_next_page,omitempty"`
}

// EventVideo points to a VOD that overlaps with an event.
type EventVideo struct {
	VodID  string `json:"vod_id"`
	Cursor string `json:"cursor,omitempty"`

	// The number of seconds from that the start of the VOD to when the event starts.
	OffsetSeconds int `json:"offset_seconds"`

	// The time that video on the event begins.  This may be after the start time of the VOD when the VOD
	// starts with video that doesn't cover the event.
	StartTime time.Time `json:"start_time"`
}

func (c *ClientImpl) GetEventVideos(ctx context.Context, eventID string, options *GetEventArchiveVideosOpts, reqOpts *twitchclient.ReqOpts) (*EventVideos, error) {
	path := "/v1/get_event_videos"
	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("event_id", eventID)

	var response EventVideos
	err = c.http(ctx, "get_event_videos", "GET", path, query, nil, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}
