// +build integration

package main

import (
	"context"
	"sync"
	"testing"
	"time"

	channelevent "code.justin.tv/twitch-events/gea/internal/channel-event"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/lib/geaclient"
	. "github.com/smartystreets/goconvey/convey"
)

// testInjectables encapsulates dependencies that are injected into the Scheduler to mock services and to help inspect
// what the Scheduler is doing.
type channelEventTestInjectables struct {
	sqsClient             *stubChannelEventSQSClient
	schedulerTimeProvider *timeProvider
	wg                    *sync.WaitGroup
}

func newChannelEventTestInjectables() *channelEventTestInjectables {
	tp := &timeProvider{}
	return &channelEventTestInjectables{
		sqsClient:             &stubChannelEventSQSClient{},
		schedulerTimeProvider: tp,
		wg:                    &sync.WaitGroup{},
	}
}

func TestCreateChannelEventUpdates(t *testing.T) {
	t.Parallel()
	now := db.ConvertToDBTime(time.Now())
	channelEventTestInjectables := newChannelEventTestInjectables()

	injectables := newDefaultInjectables()
	injectables.schedulerNowFunc = channelEventTestInjectables.schedulerTimeProvider.getTime
	injectables.channelEventSQSClient = channelEventTestInjectables.sqsClient
	injectables.createChannelEventUpdatesDoneFunc = channelEventTestInjectables.wg.Done

	ts := startServer(t, injectables, map[string][]byte{
		"gea.channel_event_window_lower_bound": []byte("5m"),
	})
	if ts == nil {
		t.Fatalf("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)

		thirtySecondsAgo := now.Add((-1) * time.Second * 30)
		tenMinutesAgo := now.Add((-1) * time.Minute * 10)
		thirtyMinutesAgo := now.Add((-1) * time.Minute * 30)
		oneHourAgo := now.Add((-1) * time.Hour * 1)

		Convey("Should queue channelIDs for events that have started and ended recently", func() {
			testDef := &testCase{
				eventProps: []*eventProperties{
					{
						startTime: thirtySecondsAgo,
					},
					{
						startTime: tenMinutesAgo,
						endTime:   &thirtySecondsAgo,
					},
				},
				timeSchedulerChecksForEvents: now,
			}
			output := runCreateChannelEventUpdates(ts, channelEventTestInjectables, testDef)

			So(output.createdEventChannelIDs, ShouldHaveLength, len(testDef.eventProps))
			So(output.enqueuedChannelIDs, ShouldNotContainDuplicates)
			for _, createdChannelID := range output.createdEventChannelIDs {
				So(output.enqueuedChannelIDs, ShouldContain, createdChannelID)
			}
		})

		Convey("Should not queue channelIDs for events that have not started or ended recently", func() {
			testDef := &testCase{
				eventProps: []*eventProperties{
					{
						startTime: oneHourAgo,
						endTime:   &thirtyMinutesAgo,
					},
				},
				timeSchedulerChecksForEvents: now,
			}
			output := runCreateChannelEventUpdates(ts, channelEventTestInjectables, testDef)

			So(output.createdEventChannelIDs, ShouldHaveLength, len(testDef.eventProps))
			So(output.enqueuedChannelIDs, ShouldNotContainDuplicates)
			for _, createdChannelID := range output.createdEventChannelIDs {
				So(output.enqueuedChannelIDs, ShouldNotContain, createdChannelID)
			}
		})

		Convey("Should not queue a channelID multiple times in successive invocations", func() {
			testDef := &testCase{
				eventProps: []*eventProperties{
					{
						startTime: thirtySecondsAgo,
					},
				},
				timeSchedulerChecksForEvents: now,
				numInvocations:               5,
			}
			output := runCreateChannelEventUpdates(ts, channelEventTestInjectables, testDef)

			So(output.createdEventChannelIDs, ShouldHaveLength, len(testDef.eventProps))
			So(output.enqueuedChannelIDs, ShouldNotContainDuplicates)
			for _, createdChannelID := range output.createdEventChannelIDs {
				So(output.enqueuedChannelIDs, ShouldContain, createdChannelID)
			}
		})

		Convey("Should queue a channelID for an updated event", func() {
			channelEventTestInjectables.sqsClient.clear()

			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			endTime := startTime.Add(time.Hour)
			title := "TestIntegration_Single title " + timestamp()
			description := "TestIntegration_Single description" + timestamp()

			event, err := ts.client.CreateSingleEvent(ts.ctx, geaclient.CreateSingleEventParams{
				OwnerID:     ownerID,
				StartTime:   startTime,
				EndTime:     endTime,
				TimeZoneID:  timeZoneAmericaLA,
				Language:    languageEN,
				Title:       title,
				Description: description,
				ChannelID:   ownerID,
				GameID:      overwatchGameID,
			}, ownerID, nil)
			So(err, ShouldBeNil)

			event, err = ts.client.UpdateSingleEvent(ts.ctx, event.GetID(), geaclient.UpdateSingleEventParams{
				EndTime: &thirtyMinutesAgo,
			}, ownerID, nil)
			So(err, ShouldBeNil)

			So(channelEventTestInjectables.sqsClient.getChannelIDs(), ShouldContain, ownerID)
		})

		Convey("Should queue a channelID for a deleted event", func() {
			channelEventTestInjectables.sqsClient.clear()

			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			endTime := startTime.Add(time.Hour)
			title := "TestIntegration_Single title " + timestamp()
			description := "TestIntegration_Single description" + timestamp()

			event, err := ts.client.CreateSingleEvent(ts.ctx, geaclient.CreateSingleEventParams{
				OwnerID:     ownerID,
				StartTime:   startTime,
				EndTime:     endTime,
				TimeZoneID:  timeZoneAmericaLA,
				Language:    languageEN,
				Title:       title,
				Description: description,
				ChannelID:   ownerID,
				GameID:      overwatchGameID,
			}, ownerID, nil)
			So(err, ShouldBeNil)

			event, err = ts.client.DeleteSingleEvent(ts.ctx, event.GetID(), ownerID, nil)
			So(err, ShouldBeNil)

			So(channelEventTestInjectables.sqsClient.getChannelIDs(), ShouldContain, ownerID)
		})
	})
}

type channelEventTestOutput struct {
	enqueuedChannelIDs     []string
	createdEventChannelIDs []string
}

// This method creates single events as outlined in testCase.eventProps, adding them to the local db.
// Then, based on testCase.numInvocations, it sends an http request to the route outlined in triggerCreateChannelEventUpdates, simulating that endpoint being hit.
func runCreateChannelEventUpdates(ts *testSetup, injectables *channelEventTestInjectables, tc *testCase) *channelEventTestOutput {
	output := &channelEventTestOutput{}

	injectables.sqsClient.clear()

	for _, eventProps := range tc.eventProps {
		createdEvent := createSingleEvent(ts, eventProps)
		output.createdEventChannelIDs = append(output.createdEventChannelIDs, createdEvent.ChannelID)
	}

	injectables.schedulerTimeProvider.setTime(tc.timeSchedulerChecksForEvents)

	numInvocations := tc.numInvocations
	if numInvocations == 0 {
		numInvocations = 1
	}
	for i := 0; i < numInvocations; i++ {
		triggerCreateChannelEventUpdates(ts)
		injectables.wg.Add(1)
	}
	injectables.wg.Wait()

	output.enqueuedChannelIDs = injectables.sqsClient.getChannelIDs()

	return output
}

func triggerCreateChannelEventUpdates(ts *testSetup) {
	err := request(ts, "POST", "/v1/create_channel_event_updates", nil, nil)
	So(err, ShouldBeNil)
}

func createSingleEvent(ts *testSetup, args *eventProperties) *geaclient.SingleEvent {
	endTime := args.startTime.Add(defaultEventLength)
	if args.endTime != nil {
		endTime = *args.endTime
	}

	channelID := timestampUser()
	title := "TestIntegration_Single title " + timestamp()
	description := "TestIntegration_Single description" + timestamp()

	event, err := ts.client.CreateSingleEvent(ts.ctx, geaclient.CreateSingleEventParams{
		OwnerID:     channelID,
		StartTime:   args.startTime,
		EndTime:     endTime,
		Language:    languageEN,
		Title:       title,
		Description: description,
		ChannelID:   channelID,
		GameID:      overwatchGameID,
	}, channelID, nil)

	So(err, ShouldBeNil)
	So(event, ShouldNotBeNil)

	return event
}

type stubChannelEventSQSClient struct {
	channelIDs []string
	rwMutex    sync.RWMutex
}

var _ channelevent.SQSClient = &stubChannelEventSQSClient{}

func (c *stubChannelEventSQSClient) SendChannelToSQS(ctx context.Context, channelID string) error {
	c.rwMutex.Lock()
	defer c.rwMutex.Unlock()

	c.channelIDs = append(c.channelIDs, channelID)
	return nil
}

func (c *stubChannelEventSQSClient) clear() {
	c.rwMutex.Lock()
	defer c.rwMutex.Unlock()

	c.channelIDs = make([]string, 0)
}

func (c *stubChannelEventSQSClient) getChannelIDs() []string {
	c.rwMutex.RLock()
	defer c.rwMutex.RUnlock()

	channelIDs := make([]string, 0, len(c.channelIDs))
	return append(channelIDs, c.channelIDs...)
}
