// +build integration

package main

import (
	"context"
	"testing"
	"time"

	"fmt"
	"sync"

	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/twitch-events/gea/internal/pubsub"
	testutils "code.justin.tv/twitch-events/gea/internal/test-utils"
	"code.justin.tv/twitch-events/gea/internal/types"
	. "github.com/smartystreets/goconvey/convey"
)

func TestChannelEventPublisher(t *testing.T) {
	t.Parallel()

	now := time.Date(2018, 01, 01, 0, 0, 0, 00, time.UTC)

	pubsubClient := &pubsubStub{}
	injectables := newDefaultInjectables()
	injectables.getChannelClient = testutils.NewGetChannelClientStub(numericalOverwatchGameID)
	injectables.clock = &testutils.StubClock{ControlledNow: now}
	injectables.pubsubClient = pubsubClient

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := context.Background()

		Convey("Should be able to publish that an event with a parent event is live", func() {
			pubsubClient.reset()
			channelID := timestampUser()

			// Create a Timetable event.
			timetableArgs := timetableParams(channelID)
			timetableEvent, err := client.CreateTimetableEvent(ctx, timetableArgs, channelID, nil)
			So(err, ShouldBeNil)
			So(timetableEvent, ShouldNotBeNil)

			// Create a Segment event within the table that is currently live.
			startTime := now.Add((-1) * time.Second)
			endTime := now.Add(time.Second)
			segmentArgs := segmentParams(timetableEvent.ID, channelID, overwatchGameID, startTime, endTime)
			segmentEvent, err := client.CreateSegmentEvent(ctx, segmentArgs, channelID, nil)
			So(err, ShouldBeNil)
			So(segmentEvent, ShouldNotBeNil)

			// Run ChannelEventPublisher's PublishChannelEvent.
			err = triggerPublishChannelEvent(ts, channelID)
			So(err, ShouldBeNil)

			// Get the messages that were sent to Pubsub.
			payloads := pubsubClient.getCurrentEvents()
			So(payloads, ShouldHaveLength, 1)
			payload := payloads[0]

			// Verify that the message had the expected content.
			So(payload, ShouldNotBeNil)
			So(payload.channelID, ShouldEqual, channelID)
			currentEvent := payload.currentEvent
			So(currentEvent, ShouldNotBeNil)
			So(currentEvent.EventID, ShouldEqual, segmentEvent.ID)
			So(currentEvent.Title, ShouldEqual, segmentEvent.Title)
			So(currentEvent.Type, ShouldEqual, segmentEvent.Type)

			parent := currentEvent.Parent
			So(parent, ShouldNotBeNil)
			So(parent.EventID, ShouldEqual, timetableEvent.ID)
			So(parent.Title, ShouldEqual, timetableEvent.Title)
			So(parent.Type, ShouldEqual, types.EventTypeTimetable)
			So(parent.Parent, ShouldBeNil)
		})

		Convey("Should be able to publish that an event with no parent is live", func() {
			pubsubClient.reset()
			channelID := timestampUser()

			// Create a Single event that is currently live.
			startTime := now.Add((-1) * time.Second)
			singleArgs := defaultCreateSingleEventParams(channelID, startTime, time.Second*2)
			singleEvent, err := client.CreateSingleEvent(ctx, singleArgs, channelID, nil)
			So(err, ShouldBeNil)
			So(singleEvent, ShouldNotBeNil)

			// Run ChannelEventPublisher's PublishChannelEvent.
			err = triggerPublishChannelEvent(ts, channelID)
			So(err, ShouldBeNil)

			// Get the messages that were sent to Pubsub.
			payloads := pubsubClient.getCurrentEvents()
			So(payloads, ShouldHaveLength, 1)
			payload := payloads[0]

			// Verify that the message had the expected content.
			So(payload, ShouldNotBeNil)
			So(payload.channelID, ShouldEqual, channelID)
			currentEvent := payload.currentEvent
			So(currentEvent, ShouldNotBeNil)
			So(currentEvent.EventID, ShouldEqual, singleEvent.ID)
			So(currentEvent.Title, ShouldEqual, singleEvent.Title)
			So(currentEvent.Type, ShouldEqual, singleEvent.Type)
			So(currentEvent.Parent, ShouldBeNil)
		})

		Convey("Should be able to publish that no event is live", func() {
			pubsubClient.reset()
			channelID := timestampUser()

			// Run ChannelEventPublisher's PublishChannelEvent.
			err := triggerPublishChannelEvent(ts, channelID)
			So(err, ShouldBeNil)

			// Get the messages that were sent to Pubsub.
			payloads := pubsubClient.getCurrentEvents()
			So(payloads, ShouldHaveLength, 1)
			payload := payloads[0]

			// Verify that the message had the expected content.
			So(payload, ShouldNotBeNil)
			So(payload.channelID, ShouldEqual, channelID)
			So(payload.currentEvent, ShouldBeNil)

			Convey("After publishing, the channel should be throttled", func() {
				throttler := ts.thisInstance.channelEventPublisher.Throttler
				So(throttler, ShouldNotBeNil)

				err := throttler.AddChannelIfNotThrottled(ctx, channelID)
				So(err, ShouldEqual, memcache.ErrNotStored)
			})
		})
	})
}

func triggerPublishChannelEvent(ts *testSetup, channelID string) error {
	url := fmt.Sprintf("/test/publish_channel_event?channel_id=%s", channelID)
	return request(ts, "POST", url, nil, nil)
}

type channelEvent struct {
	channelID    string
	currentEvent *pubsub.CurrentEvent
}

type pubsubStub struct {
	currentEvents []*channelEvent
	mutex         sync.Mutex
}

var _ pubsub.Client = &pubsubStub{}

func (p *pubsubStub) PublishCurrentChannelEvent(ctx context.Context, targetChannelID string, currentEvent *pubsub.CurrentEvent) error {
	p.mutex.Lock()
	p.currentEvents = append(p.currentEvents, &channelEvent{
		channelID:    targetChannelID,
		currentEvent: currentEvent,
	})
	p.mutex.Unlock()
	return nil
}

func (p *pubsubStub) getCurrentEvents() []*channelEvent {
	p.mutex.Lock()
	defer p.mutex.Unlock()

	currentEvents := make([]*channelEvent, 0, len(p.currentEvents))
	currentEvents = append(currentEvents, p.currentEvents...)

	return currentEvents
}

func (p *pubsubStub) reset() {
	p.mutex.Lock()
	p.currentEvents = nil
	defer p.mutex.Unlock()
}
