// +build integration

package main

import (
	"fmt"
	"testing"
	"time"

	. "github.com/smartystreets/goconvey/convey"
)

func TestHardDeleteUserWorker(t *testing.T) {
	injectables := newDefaultInjectables()

	ts := startServer(t, injectables)
	if ts == nil {
		t.Fatalf("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		ctx := ts.ctx

		Convey("Hard Delete User worker deletes all event data related to the user", func() {
			geaClient := ts.client
			dbClient := ts.thisInstance.oracleDB
			followsClient := ts.thisInstance.eventFollows

			Convey("Hard Delete User worker deletes all events created by the user and their associated data", func() {
				// Create a parent event and a child event belonging to the user.
				userID := timestampUser()
				timetableEventParams := timetableParams(userID)
				parentEvent, err := geaClient.CreateTimetableEvent(ctx, timetableEventParams, userID, nil)
				So(err, ShouldBeNil)
				So(parentEvent, ShouldNotBeNil)
				parentID := parentEvent.ID

				startTime := time.Now().UTC()
				segmentEventParams := segmentParams(parentID, userID, overwatchGameID, startTime, startTime.Add(time.Hour))
				segmentEvent, err := geaClient.CreateSegmentEvent(ctx, segmentEventParams, userID, nil)
				So(err, ShouldBeNil)
				So(segmentEvent, ShouldNotBeNil)

				// Create a single event belonging to the user.
				singleEventParams := defaultCreateSingleEventParams(userID, time.Now().UTC(), time.Hour)
				singleEvent, err := geaClient.CreateSingleEvent(ctx, singleEventParams, userID, nil)
				So(err, ShouldBeNil)
				So(singleEvent, ShouldNotBeNil)

				// Create a premiere event belonging to the user.
				premiereEventParams := defaultCreatePremiereEventParams(userID, time.Now().UTC(), time.Hour)
				premiereEvent, err := geaClient.CreatePremiereEvent(ctx, premiereEventParams, userID, nil)
				So(err, ShouldBeNil)
				So(premiereEvent, ShouldNotBeNil)

				eventIDs := []string{parentEvent.ID, segmentEvent.ID, singleEvent.ID, premiereEvent.ID}

				// Have users follow each event belonging to the user.
				for _, eventID := range eventIDs {
					follower := timestampUser()
					err = geaClient.FollowEvent(ctx, eventID, follower, nil)
					So(err, ShouldBeNil)
				}

				eventStats, err := dbClient.GetEventStats(ctx, eventIDs)
				So(err, ShouldBeNil)
				for _, stats := range eventStats {
					So(stats.FollowCount, ShouldEqual, 1)
				}

				err = triggerHardDeleteUser(ts, userID)
				So(err, ShouldBeNil)

				// check that the events no longer exist
				for _, eventID := range eventIDs {
					event, err := dbClient.GetEvent(ctx, eventID, true)
					So(err, ShouldBeNil)
					So(event, ShouldBeNil)
				}

				//check that the follows are removed
				for _, eventID := range eventIDs {
					follows, err := followsClient.GetFollowersByEventID(ctx, eventID, 10, "")
					So(err, ShouldBeNil)
					So(follows.UserIDs, ShouldBeEmpty)
					So(follows.Cursor, ShouldBeBlank)
				}
			})

			Convey("Hard Delete User worker deletes all follows of the user", func() {
				userID := timestampUser()

				// Create a parent event and a child event belonging to other users.
				timetableOwner := timestampUser()
				timetableEventParams := timetableParams(timetableOwner)
				parentEvent, err := geaClient.CreateTimetableEvent(ctx, timetableEventParams, timetableOwner, nil)
				So(err, ShouldBeNil)
				So(parentEvent, ShouldNotBeNil)
				parentID := parentEvent.ID

				startTime := time.Now().UTC()
				segmentEventParams := segmentParams(parentID, timetableOwner, overwatchGameID, startTime, startTime.Add(time.Hour))
				segmentEvent, err := geaClient.CreateSegmentEvent(ctx, segmentEventParams, timetableOwner, nil)
				So(err, ShouldBeNil)
				So(segmentEvent, ShouldNotBeNil)

				// Create a single event belonging to another user.
				singleEventOwner := timestampUser()
				singleEventParams := defaultCreateSingleEventParams(singleEventOwner, time.Now().UTC(), time.Hour)
				singleEvent, err := geaClient.CreateSingleEvent(ctx, singleEventParams, singleEventOwner, nil)
				So(err, ShouldBeNil)
				So(singleEvent, ShouldNotBeNil)

				// Create a premiere event belonging to another user.
				premiereEventOwner := timestampUser()
				premiereEventParams := defaultCreatePremiereEventParams(premiereEventOwner, time.Now().UTC(), time.Hour)
				premiereEvent, err := geaClient.CreatePremiereEvent(ctx, premiereEventParams, premiereEventOwner, nil)
				So(err, ShouldBeNil)
				So(premiereEvent, ShouldNotBeNil)

				eventIDs := []string{parentEvent.ID, segmentEvent.ID, singleEvent.ID, premiereEvent.ID}

				// Have the user follow each event belonging to other users.
				for _, eventID := range eventIDs {
					err = geaClient.FollowEvent(ctx, eventID, userID, nil)
					So(err, ShouldBeNil)
				}

				eventStats, err := dbClient.GetEventStats(ctx, eventIDs)
				So(err, ShouldBeNil)
				for _, stats := range eventStats {
					So(stats.FollowCount, ShouldEqual, 1)
				}

				err = triggerHardDeleteUser(ts, userID)
				So(err, ShouldBeNil)

				for _, eventID := range eventIDs {
					follows, err := followsClient.GetFollowersByEventID(ctx, eventID, 10, "")
					So(err, ShouldBeNil)
					So(follows.UserIDs, ShouldBeEmpty)
					So(follows.Cursor, ShouldBeBlank)
				}

				newEventStats, err := dbClient.GetEventStats(ctx, eventIDs)
				So(err, ShouldBeNil)
				for _, stats := range newEventStats {
					So(stats.FollowCount, ShouldBeZeroValue)
				}
			})

			Convey("Hard Delete User worker deletes both events and follows of the user", func() {
				userID := timestampUser()

				// Create a single event belonging to the user.
				singleEventParams := defaultCreateSingleEventParams(userID, time.Now().UTC(), time.Hour)
				singleEvent, err := geaClient.CreateSingleEvent(ctx, singleEventParams, userID, nil)
				So(err, ShouldBeNil)
				So(singleEvent, ShouldNotBeNil)

				// Create a single event belonging to another user.
				followingSingleEventOwner := timestampUser()
				followingSingleEventParams := defaultCreateSingleEventParams(followingSingleEventOwner, time.Now().UTC(), time.Hour)
				followingSingleEvent, err := geaClient.CreateSingleEvent(ctx, followingSingleEventParams, followingSingleEventOwner, nil)
				So(err, ShouldBeNil)
				So(followingSingleEvent, ShouldNotBeNil)

				// Have the user follow the single event.
				err = geaClient.FollowEvent(ctx, followingSingleEvent.ID, userID, nil)
				So(err, ShouldBeNil)

				eventStats, err := dbClient.GetEventStats(ctx, []string{followingSingleEvent.ID})
				So(err, ShouldBeNil)
				So(eventStats, ShouldHaveLength, 1)
				So(eventStats[0].FollowCount, ShouldEqual, 1)

				err = triggerHardDeleteUser(ts, userID)
				So(err, ShouldBeNil)

				// check that the user's single event no longer exist
				event, err := dbClient.GetEvent(ctx, singleEvent.ID, true)
				So(err, ShouldBeNil)
				So(event, ShouldBeNil)

				follows, err := followsClient.GetFollowersByEventID(ctx, followingSingleEvent.ID, 10, "")
				So(err, ShouldBeNil)
				So(follows.UserIDs, ShouldBeEmpty)
				So(follows.Cursor, ShouldBeBlank)

				newEventStats, err := dbClient.GetEventStats(ctx, []string{followingSingleEvent.ID})
				So(err, ShouldBeNil)
				So(newEventStats, ShouldHaveLength, 1)
				So(newEventStats[0].FollowCount, ShouldBeZeroValue)
			})

			Convey("Hard Delete User worker does not delete other users' events", func() {
				userID := timestampUser()

				// Create a single event belonging to another user.
				singleEventOwner := timestampUser()
				singleEventParams := defaultCreateSingleEventParams(singleEventOwner, time.Now().UTC(), time.Hour)
				singleEvent, err := geaClient.CreateSingleEvent(ctx, singleEventParams, singleEventOwner, nil)
				So(err, ShouldBeNil)
				So(singleEvent, ShouldNotBeNil)

				err = triggerHardDeleteUser(ts, userID)
				So(err, ShouldBeNil)

				// check that the other user's single event still exists
				event, err := dbClient.GetEvent(ctx, singleEvent.ID, true)
				So(err, ShouldBeNil)
				So(event, ShouldNotBeNil)
			})
		})
	})
}

func triggerHardDeleteUser(ts *testSetup, userID string) error {
	url := fmt.Sprintf("/test/user_hard_delete?user_id=%s", userID)
	return request(ts, "POST", url, nil, nil)
}
