package api

import (
	"context"
	"strings"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"code.justin.tv/twitch-events/gea/internal/auth"
	usersservice_client "code.justin.tv/web/users-service/client"
	"code.justin.tv/web/users-service/client/usersclient_internal"
	usersmodels "code.justin.tv/web/users-service/models"
)

// DistconfAdminList facilitates checking whether a user is allowed to perform any operation in the events
// system using a list of user IDs in Consul.
type DistconfAdminList struct {
	Config *DistconfAdminListConfig
}

var _ auth.AdminList = &DistconfAdminList{}

func (d *DistconfAdminList) IsEventsAdmin(ctx context.Context, userID string) bool {
	if userID == "" {
		return false
	}
	return strings.Contains(d.Config.adminUsers.Get(), ","+userID+",")
}

type DistconfAdminListConfig struct {
	adminUsers *distconf.Str
}

func (c *DistconfAdminListConfig) Load(d *distconf.Distconf) error {
	c.adminUsers = d.Str("gea.admin_users", "")
	return nil
}

// AuthUsersClientImpl retrieves information from the Users service that is required for authorization.
type AuthUsersClientImpl struct {
	UsersClient usersclient_internal.InternalClient
	Log         *log.ElevatedLog
}

var _ auth.Client = &AuthUsersClientImpl{}

func (i *AuthUsersClientImpl) GetUserAuthProperties(ctx context.Context, userID string) *auth.UserAuthProperties {
	filterParams := &usersmodels.FilterParams{
		NotDeleted: true,
	}
	userProps, err := i.UsersClient.GetUserByIDAndParams(ctx, userID, filterParams, nil)

	if userProps == nil || usersservice_client.IsUserNotFound(err) {
		return nil
	} else if err != nil {
		// If an error occurs when communicating with the Users service, we return properties to safely degrade
		// the auth checks that need this information, instead of returning an error for all requests.
		i.Log.LogCtx(ctx, "userID", userID, "err", err, "error getting user properties for authorization")
		return &auth.UserAuthProperties{
			IsTwitchAdmin: false,
			IsSuspended:   false,
		}
	}

	hasTOSViolation := userProps.TermsOfServiceViolation != nil && *userProps.TermsOfServiceViolation
	hasDmcaViolation := userProps.DmcaViolation != nil && *userProps.DmcaViolation

	return &auth.UserAuthProperties{
		IsTwitchAdmin: userProps.Admin != nil && *userProps.Admin,
		IsSuspended:   hasTOSViolation || hasDmcaViolation,
	}
}
