package api

import (
	"bytes"
	"context"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"math"
	"net/http"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/spade"
	"code.justin.tv/twitch-events/gea/cmd/gea/internal/api/models"
	"code.justin.tv/twitch-events/gea/internal/auth"
	channelevent "code.justin.tv/twitch-events/gea/internal/channel-event"
	"code.justin.tv/twitch-events/gea/internal/clock"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/follows"
	"code.justin.tv/twitch-events/gea/internal/hallpass"
	"code.justin.tv/twitch-events/gea/internal/hypeman"
	hypemanscheduler "code.justin.tv/twitch-events/gea/internal/hypeman-scheduler"
	hypemanworker "code.justin.tv/twitch-events/gea/internal/hypeman-worker"
	"code.justin.tv/twitch-events/gea/internal/images"
	jax "code.justin.tv/twitch-events/gea/internal/jax-client"
	"code.justin.tv/twitch-events/gea/internal/types"
	harddelete "code.justin.tv/twitch-events/gea/internal/user-hard-delete"
	"code.justin.tv/twitch-events/gea/internal/video"
	"code.justin.tv/twitch-events/gea/lib/geaclient"

	goji "goji.io"
	"goji.io/pat"
)

const apiTimeout = 5 * time.Second

type HTTPServer struct {
	service_common.BaseHTTPServer
	Config *HTTPConfig

	EventFollows          *follows.EventFollows
	EventHandlers         *types.EventHandlers
	HypemanScheduler      *hypemanscheduler.Scheduler
	HypemanWorker         *hypemanworker.Worker
	UserHardDeleteWorker  *harddelete.Worker
	ImageUploader         *images.ImageUploadService
	JaxClient             jax.Client
	ChannelEventScheduler *channelevent.Scheduler
	HallpassClient        hallpass.Client
	OracleDB              db.DB
	SpadeClient           SpadeClient
	SuggestionsConfig     *SuggestionsConfig
	Clock                 clock.Clock
	LiveEventLoader       *channelevent.LiveEventLoader
	ChannelEventPublisher *channelevent.Publisher
	AdminList             auth.AdminList
	AuthUsersClient       auth.Client
}

type HTTPConfig struct {
	service_common.BaseHTTPServerConfig

	canCreateEventsOnAnyChannelUsers *distconf.Str
	logAllUpdateEventErrors          *distconf.Bool
}

func (c *HTTPConfig) Load(d *distconf.Distconf) error {
	if err := c.BaseHTTPServerConfig.Verify(d, "gea"); err != nil {
		return err
	}
	c.canCreateEventsOnAnyChannelUsers = d.Str("gea.can_create_events_on_any_channel_users", "")
	c.logAllUpdateEventErrors = d.Bool("gea.log_all_update_event_errors", false)
	return nil
}

// SpadeClient serves as an interface for mocking the spade client
type SpadeClient interface {
	QueueEvents(events ...spade.Event)
	Start() error
}

func (s *HTTPServer) Setup() error {
	err := s.BaseHTTPServer.Setup()
	if err != nil {
		return err
	}
	s.SuggestionsConfig.configByGame.Watch(s.loadConfigByGame)
	s.loadConfigByGame()
	return nil
}

//TODO: move this to service-common/http
type cachedResponse struct {
	payload interface{}

	cacheTime time.Duration
}

func (c *cachedResponse) Headers() map[string]string {
	cacheTime := int(math.Ceil(c.cacheTime.Seconds()))
	if cacheTime == 0 {
		return nil
	}
	return map[string]string{
		"Cache-Control": fmt.Sprintf("public, max-age=%d", cacheTime),
	}
}

func (c cachedResponse) MarshalJSON() ([]byte, error) {
	return json.Marshal(c.payload)
}

// Wraps an http.Handler with a request timeout.
func timeoutHandler(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		timeoutCtx, cancel := context.WithTimeout(r.Context(), apiTimeout)
		defer cancel()
		r = r.WithContext(timeoutCtx)
		h.ServeHTTP(w, r)
	})
}

func (s *HTTPServer) Routes(mux *goji.Mux) {
	/*
	   TODO: Routes

	   Get Available Broadcasts for Channel
	   Update Event Node
	   Get Available Broadcasts for Game

	   Follow Event
	   Get Follow State For User
	   Delete Follow

	   Delete Event Node
	*/

	mux.Use(timeoutHandler)

	mux.Handle(pat.Get("/v1/get_event"), s.CreateHandler("v1_get_event", s.getEvent))
	mux.Handle(pat.Get("/v1/get_events"), s.CreateHandler("v1_get_events", s.getEvents))
	mux.Handle(pat.Get("/v1/get_live_event"), s.CreateHandler("v1_get_live_event", s.getLiveEvent))
	mux.Handle(pat.Get("/v1/get_event_metadata"), s.CreateHandler("v1_get_event_metadata", s.getEventMetadata))

	mux.Handle(pat.Post("/v1/create_event"), s.CreateHandler("v1_create_event", s.createEvent))
	mux.Handle(pat.Post("/v1/delete_event"), s.CreateHandler("v1_delete_event", s.deleteEvent))
	mux.Handle(pat.Post("/v1/update_event"), s.CreateHandler("v1_update_event", s.updateEventAndLogError))

	mux.Handle(pat.Get("/v1/get_event_ids_by_channel_ids"), s.CreateHandler("v1_get_event_ids_by_channel_ids", s.getEventIDsByChannelIDs))
	mux.Handle(pat.Get("/v1/get_event_ids_by_types"), s.CreateHandler("v1_get_event_ids_by_types", s.getEventIDsByTypes))
	mux.Handle(pat.Get("/v1/get_event_ids_by_filter"), s.CreateHandler("v1_get_event_ids_by_filter", s.getEventIDsByFilter))
	mux.Handle(pat.Get("/v1/get_event_suggestions_by_game"), s.CreateHandler("v1_get_event_suggestions_by_game", s.getEventSuggestionsByGame))

	mux.Handle(pat.Get("/v1/get_managed_collection_ids_by_owner"), s.CreateHandler("v1_get_managed_collection_ids_by_owner", s.getManagedCollectionIDsByOwner))
	mux.Handle(pat.Get("/v1/get_managed_leaf_ids_by_owner"), s.CreateHandler("v1_get_managed_leaf_ids_by_owner", s.getManagedLeafEventIDsByOwner))
	mux.Handle(pat.Get("/v1/get_managed_leaf_ids_by_parent"), s.CreateHandler("v1_get_managed_leaf_ids_by_parent", s.getManagedLeafEventIDsByParent))

	mux.Handle(pat.Get("/v1/get_event_stats"), s.CreateHandler("v1_get_event_stats", s.getEventStats))

	mux.Handle(pat.Get("/v1/get_event_videos"), s.CreateHandler("v1_get_event_videos", s.getEventVideos))

	mux.Handle(pat.Post("/v1/add_localization"), s.CreateHandler("v1_add_localization", s.addLocalization))
	mux.Handle(pat.Post("/v1/update_localization"), s.CreateHandler("v1_update_localization", s.updateLocalization))
	mux.Handle(pat.Post("/v1/remove_localization"), s.CreateHandler("v1_remove_localization", s.removeLocalization))

	mux.Handle(pat.Post("/v1/follow_event"), s.CreateHandler("v1_follow_event", s.followEvent))
	mux.Handle(pat.Post("/v1/unfollow_event"), s.CreateHandler("v1_unfollow_event", s.unfollowEvent))
	mux.Handle(pat.Get("/v1/get_event_followers"), s.CreateHandler("v1_get_event_followers", s.getEventFollowers))
	mux.Handle(pat.Get("/v1/get_followed_events"), s.CreateHandler("v1_get_followed_events", s.getFollowedEvents))
	mux.Handle(pat.Get("/v1/check_followed_events"), s.CreateHandler("v1_check_followed_events", s.checkFollowedEvents))

	mux.Handle(pat.Post("/v1/reserve_image_upload"), s.CreateHandler("v1_reserve_image_upload", s.reserveImageUpload))
	mux.Handle(pat.Post("/v1/send_event_started_notifications"), s.CreateHandler("v1_send_event_started_notifications", s.sendEventStartedNotifications))

	// TODO: This should be deleted this is only so we can do a drop in replacement for oracle uploads
	mux.Handle(pat.Post("/v1/legacy_image_upload"), s.CreateHandler("v1_legacy_image_upload", s.legacyImageUpload))

	// v1/create_notification_jobs is a temporary endpoint to find events that are starting, and enqueue notification
	// jobs for them.  This should be removed when notification creation is moved to its own service.
	mux.Handle(pat.Post("/v1/create_notification_jobs"), s.CreateHandler("v1_create_notification_jobs", s.createNotificationJobs))

	mux.Handle(pat.Post("/v1/create_channel_event_updates"), s.CreateHandler("v1_create_channel_event_updates", s.createChannelEventUpdates))

	mux.Handle(pat.Post("/test/process_notification_job"), s.CreateHandler("test_process_notification_job", s.processNotificationJob))
	mux.Handle(pat.Post("/test/publish_channel_event"), s.CreateHandler("test_publish_channel_event", s.publishChannelEvent))
	mux.Handle(pat.Post("/test/user_hard_delete"), s.CreateHandler("test_user_hard_delete", s.processHardDeleteUserJob))
}

func requireUserID(r *http.Request) (string, error) {
	userID := r.URL.Query().Get("user_id")
	if userID == "" {
		return "", &service_common.CodedError{
			Code: http.StatusForbidden,
			Err:  errors.New("you must provide a user_id"),
		}
	}
	return userID, nil
}

func requireString(r *http.Request, paramName string) (string, error) {
	str := r.URL.Query().Get(paramName)
	if str == "" {
		return "", &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("you must provide a " + paramName),
		}
	}
	return str, nil
}

func requireStrings(r *http.Request, paramName string) ([]string, error) {
	parts := strings.Split(r.URL.Query().Get(paramName), ",")
	values := make([]string, 0, len(parts))
	for _, part := range parts {
		value := strings.TrimSpace(part)
		if value != "" {
			values = append(values, value)
		}
	}
	if len(values) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("you must provide " + paramName),
		}
	}
	return values, nil
}

func requestOptionalStrings(r *http.Request, paramName string) ([]string, error) {
	str := r.URL.Query().Get(paramName)
	if str == "" {
		return nil, nil
	}
	return requireStrings(r, paramName)
}

func requestOptionalStringFromSet(r *http.Request, paramName string, validSet []string, defaultValue string) (string, error) {
	str := r.URL.Query().Get(paramName)
	str = strings.TrimSpace(str)
	if str == "" {
		return defaultValue, nil
	}

	for _, validValue := range validSet {
		if str == validValue {
			return str, nil
		}
	}

	return "", &service_common.CodedError{
		Code: http.StatusBadRequest,
		Err:  errors.Errorf("param %s was expected to be in set, \"%s\"", paramName, strings.Join(validSet, ", ")),
	}
}

func requestBool(r *http.Request, paramName string, defaultValue bool) (bool, error) {
	str := r.URL.Query().Get(paramName)
	if str == "" {
		return defaultValue, nil
	}
	if str != "true" && str != "false" {
		return false, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New(paramName + " must be 'true' or 'false'"),
		}
	}
	return str == "true", nil
}

func requestOptionalTime(r *http.Request, paramName string) (*time.Time, error) {
	str := r.URL.Query().Get(paramName)
	if str == "" {
		return nil, nil
	}

	t, err := time.Parse(time.RFC3339, str)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("you must provide a valid " + paramName),
		}
	}
	return &t, nil
}

func requestInt(r *http.Request, paramName string, defaultValue int) (int, error) {
	intParam := r.URL.Query().Get(paramName)
	if intParam == "" {
		return defaultValue, nil
	}
	asInt, err := strconv.Atoi(intParam)
	if err != nil {
		return 0, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, fmt.Sprintf("expect integer param, saw %s", intParam)),
		}
	}
	return asInt, nil
}

func dedupeIDs(ids []string) []string {
	d := make([]string, 0, len(ids))
	dupe := make(map[string]bool)
	for _, id := range ids {
		id = strings.TrimSpace(id)
		if id != "" && !dupe[id] {
			dupe[id] = true
			d = append(d, id)
		}
	}
	return d
}

func getPagination(r *http.Request, defaultLimit, maxLimit int) (int, string, error) {
	limit, err := requestInt(r, "limit", defaultLimit)
	if err != nil {
		return 0, "", err
	}
	if limit > maxLimit {
		return 0, "", &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("limit cannot be higher than %v", maxLimit),
		}
	}
	cursor := r.URL.Query().Get("cursor")
	return limit, cursor, nil
}

func (s *HTTPServer) requireDBEvent(ctx context.Context, eventID string) (*db.Event, error) {
	if eventID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("event id is required"),
		}
	}

	dbEvent, err := s.OracleDB.GetEvent(ctx, eventID, false)
	if err != nil {
		return nil, err
	}
	if dbEvent == nil || dbEvent.DeletedAt != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("event not found"),
		}
	}

	return dbEvent, nil
}

func (s *HTTPServer) createEvent(r *http.Request) (interface{}, error) {
	ctx := r.Context()
	userID, err := requireUserID(r)
	if err != nil {
		return nil, err
	}

	var params models.V1CreateEventParams
	if err = json.NewDecoder(r.Body).Decode(&params); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "could not parse body"),
		}
	}
	if params.OwnerID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("you must provide an owner_id"),
		}
	}

	if err = s.canCreateEvent(ctx, userID, params.OwnerID, params.ChannelID); err != nil {
		return nil, err
	}

	now := s.Clock.NowUTC()

	var imageID *images.ImageID
	if params.CoverImageID != nil {
		switch params.CoverImageID.Type {
		case "gea":
			imageID, err = s.ImageUploader.SaveUploadedImage(r.Context(), params.CoverImageID.ID)
			if err != nil {
				return nil, err
			}
		default:
			imageID = params.CoverImageID
		}
	} else {
		// Pick default image
		imageID, err = s.ImageUploader.PickDefaultImage(r.Context())
		if err != nil {
			return nil, err
		}
	}

	event, err := s.EventHandlers.CreateEvent(r.Context(), &types.EventUpdateParams{
		OwnerID:  params.OwnerID,
		Type:     params.Type,
		ParentID: params.ParentID,

		StartTime:  params.StartTime,
		EndTime:    params.EndTime,
		TimeZoneID: params.TimeZoneID,

		CoverImageID: imageID,
		Language:     params.Language,
		Title:        params.Title,
		Description:  params.Description,
		ChannelID:    params.ChannelID,
		GameID:       params.GameID,

		Timestamp: now,

		PremiereID: params.PremiereID,
	})
	if err != nil {
		if vErr, ok := err.(types.ValidationError); ok {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  vErr,
			}
		}
		return nil, err
	}

	eventTrackingObject := s.paramsToCreateEventTracking(ctx, userID, event.GetID(), now, params)
	s.SpadeClient.QueueEvents(spade.Event{
		Name:       "oracle_event_server",
		Properties: *eventTrackingObject,
	})

	return event, nil
}

func (s *HTTPServer) updateEventAndLogError(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	userID, err := requireUserID(r)
	if err != nil {
		return nil, err
	}

	var params models.V1UpdateEventParams
	if err = json.NewDecoder(r.Body).Decode(&params); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "could not parse body"),
		}
	}

	resp, err := s.updateEvent(ctx, params, userID)
	if err != nil && s.Config.logAllUpdateEventErrors.Get() {
		s.Log.LogCtx(
			ctx,
			"msg", "updating event failed",
			"err", err,
			"user_id", userID,
			"event_id", params.ID)
	}

	return resp, err
}

func (s *HTTPServer) updateEvent(ctx context.Context, params models.V1UpdateEventParams, userID string) (interface{}, error) {
	dbEvent, err := s.requireDBEvent(ctx, params.ID)
	if err != nil {
		return nil, err
	}

	canUpdateParams := &canUpdateEventsParams{
		currentOwnerID:   dbEvent.OwnerID,
		currentChannelID: dbEvent.ChannelID,

		newChannelID: params.ChannelID,
		newOwnerID:   params.OwnerID,
	}
	if err = s.canUpdateEvent(ctx, userID, canUpdateParams); err != nil {
		return nil, err
	}

	var imageID *images.ImageID
	if params.CoverImageID != nil {
		switch params.CoverImageID.Type {
		case "gea":
			if dbEvent.CoverImageID == nil || dbEvent.CoverImageID.ID != params.CoverImageID.ID {
				imageID, err = s.ImageUploader.SaveUploadedImage(ctx, params.CoverImageID.ID)
				if err != nil {
					return nil, err
				}
			} else {
				imageID = params.CoverImageID
			}

		default:
			imageID = params.CoverImageID
		}
	} else {
		imageID = dbEvent.CoverImageID
	}

	dbAttributes, err := s.OracleDB.GetEventAttributes(ctx, params.ID)
	if err != nil {
		return nil, err
	}

	params.MergeWithDB(dbEvent, dbAttributes)

	now := s.Clock.NowUTC()
	event, err := s.EventHandlers.UpdateEvent(ctx, dbEvent, &types.EventUpdateParams{
		OwnerID:  *params.OwnerID,
		Type:     *params.Type,
		ParentID: params.ParentID,

		StartTime:  params.StartTime,
		EndTime:    params.EndTime,
		TimeZoneID: params.TimeZoneID,

		CoverImageID: imageID,
		Language:     params.Language,
		Title:        params.Title,
		Description:  params.Description,
		ChannelID:    params.ChannelID,
		GameID:       params.GameID,

		Timestamp: now,

		PremiereID: params.PremiereID,
	})
	if err != nil {
		if vErr, ok := err.(types.ValidationError); ok {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  vErr,
			}
		}
		return nil, err
	}

	channelUpdateIDs := event.GetChannelIDs()
	if params.ChannelID != nil {
		channelUpdateIDs = append(channelUpdateIDs, *params.ChannelID)
	}
	s.ChannelEventScheduler.EnqueueChannelIDs(ctx, channelUpdateIDs, true)

	eventTrackingObject := s.paramsToUpdateEventTracking(ctx, userID, event.GetID(), dbEvent.GameID, dbEvent.CreatedAt, now, params)
	s.SpadeClient.QueueEvents(spade.Event{
		Name:       "oracle_event_server",
		Properties: *eventTrackingObject,
	})

	return event, nil
}

func (s *HTTPServer) deleteEvent(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	userID, err := requireUserID(r)
	if err != nil {
		return nil, err
	}

	var params models.V1DeleteEventParams
	if err = json.NewDecoder(r.Body).Decode(&params); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "could not parse body"),
		}
	}

	dbEvent, err := s.requireDBEvent(ctx, params.ID)
	if err != nil {
		return nil, err
	}
	if err = s.canDeleteEvent(ctx, userID, dbEvent.OwnerID, dbEvent.ChannelID); err != nil {
		return nil, err
	}

	now := s.Clock.NowUTC()

	event, err := s.EventHandlers.DeleteEvent(ctx, dbEvent, &types.EventDeleteParams{
		Timestamp: now,
	})
	if err != nil {
		if vErr, ok := err.(types.ValidationError); ok {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  vErr,
			}
		}
		return nil, err
	}

	s.ChannelEventScheduler.EnqueueChannelIDs(ctx, event.GetChannelIDs(), true)

	eventTrackingObject := s.dbEventToDeleteEventTracking(ctx, userID, event.GetID(), now, dbEvent)
	s.SpadeClient.QueueEvents(spade.Event{
		Name:       "oracle_event_server",
		Properties: *eventTrackingObject,
	})

	return event, nil
}

func (s *HTTPServer) getEvent(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	eventID, err := requireString(r, "id")
	if err != nil {
		return nil, err
	}

	getDeleted, err := requestBool(r, "deleted", false)
	if err != nil {
		return nil, err
	}
	skipCache, err := requestBool(r, "skip_cache", false)
	if err != nil {
		return nil, err
	}

	events, err := s.EventHandlers.GetEvents(ctx, []string{eventID}, getDeleted, skipCache)
	if err != nil {
		return nil, err
	}

	if len(events) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("event not found"),
		}
	}
	return events[0], nil
}

func (s *HTTPServer) getEvents(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	eventIDs, err := requireStrings(r, "ids")
	if err != nil {
		return nil, err
	}
	getDeleted, err := requestBool(r, "deleted", false)
	if err != nil {
		return nil, err
	}
	skipCache, err := requestBool(r, "skip_cache", false)
	if err != nil {
		return nil, err
	}

	events, err := s.EventHandlers.GetEvents(ctx, eventIDs, getDeleted, skipCache)
	if err != nil {
		return nil, err
	}

	return events, nil
}

func (s *HTTPServer) getLiveEvent(r *http.Request) (interface{}, error) {
	ctx := r.Context()
	cacheTime := 2 * time.Minute

	channelID, err := requireString(r, "channel_id")
	if err != nil {
		return nil, err
	}
	getDeleted, err := requestBool(r, "deleted", false)
	if err != nil {
		return nil, err
	}
	skipCache, err := requestBool(r, "skip_cache", false)
	if err != nil {
		return nil, err
	}

	liveEvent, err := s.LiveEventLoader.GetLiveEvent(ctx, channelID, getDeleted, skipCache)
	if err != nil {
		return nil, err
	}

	return &cachedResponse{
		payload:   liveEvent,
		cacheTime: cacheTime,
	}, nil
}

func (s *HTTPServer) getEventMetadata(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	eventID, err := requireString(r, "id")
	if err != nil {
		return nil, err
	}
	skipCache, err := requestBool(r, "skip_cache", false)
	if err != nil {
		return nil, err
	}

	event, err := s.EventHandlers.GetEventMetadata(ctx, eventID, skipCache)
	if err != nil {
		return nil, err
	}

	if event == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("event not found"),
		}
	}
	return event, nil
}

func (s *HTTPServer) getEventStats(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	eventIDs, err := requireStrings(r, "event_ids")
	if err != nil {
		return nil, err
	}
	eventIDs = dedupeIDs(eventIDs)

	dbStats, err := s.OracleDB.GetEventStats(ctx, eventIDs)
	if err != nil {
		return nil, err
	}

	res := &models.V1EventStatsResponse{
		Items: make([]*models.V1EventStats, 0, len(dbStats)),
	}

	for _, stats := range dbStats {
		res.Items = append(res.Items, &models.V1EventStats{
			EventID:     stats.EventID,
			FollowCount: stats.FollowCount,
		})
	}

	return res, nil
}

type getEventArchiveVideosParams struct {
	eventID  string
	pageArgs *video.GetPageArgs
}

func (s *HTTPServer) getEventVideos(r *http.Request) (interface{}, error) {
	ctx := r.Context()
	requestParams, err := s.parseGetEventArchiveVideosParams(r)
	if err != nil {
		return nil, err
	}

	// Load all VODs that are associated with the event from memcache or from Vodapi.
	videoItems, err := s.EventHandlers.GetArchiveVideosForEvent(ctx, requestParams.eventID)
	if err != nil {
		return nil, err
	}

	// Filter and sort the videos, and grab a page out of the resulting list.
	videosPage, err := video.GetPageOfVideos(videoItems, requestParams.pageArgs)
	if err != nil {
		return nil, err
	}

	// We may want to reconsider introducing a database to store event videos so that we can
	// run queries against it, instead of doing filters and sorts in memory.

	return &cachedResponse{
		payload:   convertVideosPageToV1EventVideos(videosPage),
		cacheTime: 2 * time.Minute,
	}, nil
}

func convertVideosPageToV1EventVideos(page *video.VideosPage) *models.V1EventVideos {
	videos := make([]*models.V1EventVideo, len(page.Items))
	for i, item := range page.Items {
		videos[i] = convertVideosPageItemToV1EventVideos(&item)
	}

	return &models.V1EventVideos{
		Items:       videos,
		HasNextPage: page.HasNextPage,
	}
}

func convertVideosPageItemToV1EventVideos(item *video.VideosPageItem) *models.V1EventVideo {
	return &models.V1EventVideo{
		VodID:         item.Video.VodID,
		OffsetSeconds: item.Video.OffsetSeconds,
		StartTime:     item.Video.StartTime,
		Cursor:        item.Cursor,
	}
}

func (s *HTTPServer) parseGetEventArchiveVideosParams(r *http.Request) (*getEventArchiveVideosParams, error) {
	eventID, err := requireString(r, "event_id")
	if err != nil {
		return nil, err
	}

	validSortByValues := []string{geaclient.VideoSortByStartTime, geaclient.VideoSortByViews}
	sortByParam, err := requestOptionalStringFromSet(r, "sort_by", validSortByValues, geaclient.VideoSortByStartTime)
	if err != nil {
		return nil, err
	}
	sortBy := video.SortByStartTime
	if sortByParam == geaclient.VideoSortByViews {
		sortBy = video.SortByViews
	}

	limit, err := requestInt(r, "limit", 20)
	if err != nil {
		return nil, err
	}
	if limit > 100 {
		limit = 100
	}

	afterCursor := r.URL.Query().Get("after_cursor")

	validVideoTypes := []string{geaclient.VideoTypeArchive, geaclient.VideoTypeHighlight}
	videoTypeParam, err := requestOptionalStringFromSet(r, "video_type", validVideoTypes, "")
	if err != nil {
		return nil, err
	}

	broadcastTypeFilter := ""
	switch videoTypeParam {
	case geaclient.VideoTypeArchive:
		broadcastTypeFilter = video.BroadcastTypeArchive
	case geaclient.VideoTypeHighlight:
		broadcastTypeFilter = video.BroadcastTypeHighlight
	}

	return &getEventArchiveVideosParams{
		eventID: eventID,
		pageArgs: &video.GetPageArgs{
			AfterCursor:           afterCursor,
			Limit:                 limit,
			SortBy:                sortBy,
			FilterByBroadcastType: broadcastTypeFilter,
		},
	}, nil
}

type getEventIDsByChannelIDsParams struct {
	channelIDs      []string
	startTimeAfter  *time.Time
	startTimeBefore *time.Time
	endTimeAfter    *time.Time
	endTimeBefore   *time.Time
	limit           int
	cursor          string
	desc            string
}

func (s *HTTPServer) getEventIDsByChannelIDs(r *http.Request) (interface{}, error) {
	ctx := r.Context()
	params, err := s.parseGetEventIDsByChannelIDsParams(r)
	if err != nil {
		return nil, err
	}
	noCache, err := requestBool(r, "no_cache", false)
	if err != nil {
		return nil, err
	}

	if params.startTimeAfter == nil && params.endTimeAfter == nil && params.startTimeBefore == nil && params.endTimeBefore == nil {
		now := s.Clock.NowUTC()
		params.startTimeAfter = &now
	}

	filter := &db.BroadcastFilter{
		ChannelIDs: params.channelIDs,
		StartTimeWindow: &db.TimeWindow{
			Start: params.startTimeAfter,
			End:   params.startTimeBefore,
		},
		EndTimeWindow: &db.TimeWindow{
			Start: params.endTimeAfter,
			End:   params.endTimeBefore,
		},
	}
	eventIDs, err := s.OracleDB.GetEventIDsSortedByStartTime(ctx, filter, params.desc == "true", params.cursor, params.limit)
	if err != nil {
		return nil, err
	}

	cacheTime := 2 * time.Minute
	if noCache {
		cacheTime = 0
	}

	return &cachedResponse{
		payload: &models.V1EventIDs{
			EventIDs: eventIDs.EventIDs,
			Cursor:   eventIDs.Cursor,
		},
		cacheTime: cacheTime,
	}, nil
}

func (s *HTTPServer) parseGetEventIDsByChannelIDsParams(r *http.Request) (*getEventIDsByChannelIDsParams, error) {
	channelIDs, err := requireStrings(r, "channel_ids")
	if err != nil {
		return nil, err
	}

	startTimeAfter, err := requestOptionalTime(r, "start_time_after")
	if err != nil {
		return nil, err
	}

	startTimeBefore, err := requestOptionalTime(r, "start_time_before")
	if err != nil {
		return nil, err
	}

	endTimeAfter, err := requestOptionalTime(r, "end_time_after")
	if err != nil {
		return nil, err
	}

	endTimeBefore, err := requestOptionalTime(r, "end_time_before")
	if err != nil {
		return nil, err
	}

	limit, err := requestInt(r, "limit", 200)
	if err != nil {
		return nil, err
	}
	if limit > 200 {
		limit = 200
	}

	cursor := r.URL.Query().Get("cursor")
	desc := r.URL.Query().Get("desc")

	// Default to descending for past events to emulate oracle behavior
	if desc == "" && endTimeBefore != nil {
		desc = "true"
	}

	return &getEventIDsByChannelIDsParams{
		channelIDs:      channelIDs,
		startTimeAfter:  startTimeAfter,
		startTimeBefore: startTimeBefore,
		endTimeAfter:    endTimeAfter,
		endTimeBefore:   endTimeBefore,
		limit:           limit,
		cursor:          cursor,
		desc:            desc,
	}, nil
}

type getEventIDsByTypesParams struct {
	types           []string
	startTimeAfter  *time.Time
	startTimeBefore *time.Time
	endTimeAfter    *time.Time
	endTimeBefore   *time.Time
	limit           int
	cursor          string
}

func (s *HTTPServer) getEventIDsByTypes(r *http.Request) (interface{}, error) {
	ctx := r.Context()
	params, err := s.parseGetEventIDsByTypesParams(r)
	if err != nil {
		return nil, err
	}

	filter := &db.BroadcastFilter{
		Types: params.types,
		StartTimeWindow: &db.TimeWindow{
			Start: params.startTimeAfter,
			End:   params.startTimeBefore,
		},
		EndTimeWindow: &db.TimeWindow{
			Start: params.endTimeAfter,
			End:   params.endTimeBefore,
		},
	}
	eventIDs, err := s.OracleDB.GetEventIDsSortedByID(ctx, filter, params.cursor, params.limit)
	if err != nil {
		return nil, err
	}

	return &cachedResponse{
		payload: &models.V1EventIDs{
			EventIDs: eventIDs.EventIDs,
			Cursor:   eventIDs.Cursor,
		},
		cacheTime: 2 * time.Minute,
	}, nil
}

func (s *HTTPServer) parseGetEventIDsByTypesParams(r *http.Request) (*getEventIDsByTypesParams, error) {
	types, err := requireStrings(r, "types")
	if err != nil {
		return nil, err
	}

	startTimeAfter, err := requestOptionalTime(r, "start_time_after")
	if err != nil {
		return nil, err
	}

	startTimeBefore, err := requestOptionalTime(r, "start_time_before")
	if err != nil {
		return nil, err
	}

	endTimeAfter, err := requestOptionalTime(r, "end_time_after")
	if err != nil {
		return nil, err
	}

	endTimeBefore, err := requestOptionalTime(r, "end_time_before")
	if err != nil {
		return nil, err
	}

	limit, err := requestInt(r, "limit", 20)
	if err != nil {
		return nil, err
	}
	if limit > 100 {
		limit = 100
	}

	cursor := r.URL.Query().Get("cursor")

	return &getEventIDsByTypesParams{
		types:           types,
		startTimeAfter:  startTimeAfter,
		startTimeBefore: startTimeBefore,
		endTimeAfter:    endTimeAfter,
		endTimeBefore:   endTimeBefore,
		limit:           limit,
		cursor:          cursor,
	}, nil
}

func (s *HTTPServer) getEventIDsByFilter(r *http.Request) (interface{}, error) {
	ctx := r.Context()
	requestParams, err := s.parseGetEIDsFilterOpts(r)
	if err != nil {
		return nil, err
	}

	return s.getEventIDsByFilterParams(ctx, requestParams)
}

func (s *HTTPServer) getEventIDsByFilterParams(ctx context.Context, params *getEIDsFilterOpts) (interface{}, error) {
	// We are starting to support paging by a new cursor, starting lists from an event ID, and will support starting
	// lists from an end time, and backwards pagination soon.  We limit this functionality to requests for event IDs
	// in ascending order by start time, to focus development efforts.
	if (params.sortBy == "" || params.sortBy == "start_time") && !params.desc {
		return s.getEventIDsOrderedByAscStartTime(ctx, params)
	}

	return s.getEventIDsByFilterLegacy(ctx, params)
}

type getEIDsFilterOpts struct {
	parentEventIDs []string
	types          []string
	gameIDs        []string
	channelIDs     []string
	ownerIDs       []string

	startTimeAfter  *time.Time
	startTimeBefore *time.Time
	endTimeAfter    *time.Time
	endTimeBefore   *time.Time

	limit  int
	cursor string
	sortBy string

	firstPageOptions []firstPageOption

	getPreviousPage bool
	desc            bool
	noCache         bool
}

func (s *HTTPServer) parseGetEIDsFilterOpts(r *http.Request) (*getEIDsFilterOpts, error) {
	parentEventIDs, err := requestOptionalStrings(r, "parents")
	if err != nil {
		return nil, err
	}

	types, err := requestOptionalStrings(r, "types")
	if err != nil {
		return nil, err
	}

	gameIDs, err := requestOptionalStrings(r, "game_ids")
	if err != nil {
		return nil, err
	}

	channelIDs, err := requestOptionalStrings(r, "channel_ids")
	if err != nil {
		return nil, err
	}

	ownerIDs, err := requestOptionalStrings(r, "owner_ids")
	if err != nil {
		return nil, err
	}

	startTimeAfter, err := requestOptionalTime(r, "start_time_after")
	if err != nil {
		return nil, err
	}

	startTimeBefore, err := requestOptionalTime(r, "start_time_before")
	if err != nil {
		return nil, err
	}

	endTimeAfter, err := requestOptionalTime(r, "end_time_after")
	if err != nil {
		return nil, err
	}

	endTimeBefore, err := requestOptionalTime(r, "end_time_before")
	if err != nil {
		return nil, err
	}

	limit, err := requestInt(r, "limit", 20)
	if err != nil {
		return nil, err
	}
	if limit > 100 {
		limit = 100
	}

	cursor := r.URL.Query().Get("cursor")

	sortBy, err := requestOptionalStringFromSet(r, "sort_by", []string{"hype", "start_time"}, "start_time")
	if err != nil {
		return nil, err
	}

	descending, err := requestBool(r, "desc", false)
	if err != nil {
		return nil, err
	}

	firstPageOptionStrings, err := requestOptionalStrings(r, "first_page_options")
	if err != nil {
		return nil, err
	}
	firstPageOptions := make([]firstPageOption, 0, len(firstPageOptionStrings))
	for _, firstPageOptionString := range firstPageOptionStrings {
		firstPageOption, innerErr := parseFirstPageOption(firstPageOptionString)
		if innerErr != nil {
			return nil, innerErr
		}
		if firstPageOption != nil {
			firstPageOptions = append(firstPageOptions, *firstPageOption)
		}
	}

	getPreviousPage, err := requestBool(r, "get_prev_page", false)
	if err != nil {
		return nil, err
	}
	if getPreviousPage && cursor == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("get_prev_page was true but no cursor was given"),
		}
	}

	noCache, err := requestBool(r, "no_cache", false)
	if err != nil {
		return nil, err
	}

	return &getEIDsFilterOpts{
		parentEventIDs:   parentEventIDs,
		types:            types,
		gameIDs:          gameIDs,
		channelIDs:       channelIDs,
		ownerIDs:         ownerIDs,
		startTimeAfter:   startTimeAfter,
		startTimeBefore:  startTimeBefore,
		endTimeAfter:     endTimeAfter,
		endTimeBefore:    endTimeBefore,
		limit:            limit,
		cursor:           cursor,
		sortBy:           sortBy,
		desc:             descending,
		getPreviousPage:  getPreviousPage,
		firstPageOptions: firstPageOptions,
		noCache:          noCache,
	}, nil
}

type firstPageOption struct {
	eventID   string
	endsAfter *time.Time
}

const (
	firstPagePrefixEventID   = "id"
	firstPagePrefixEndsAfter = "ends_after"
)

func parseFirstPageOption(str string) (*firstPageOption, error) {
	if str == "" {
		return nil, nil
	}

	tokens := strings.SplitN(str, ":", 2)
	if len(tokens) != 2 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("firstPageOption string \"%s\" was expected to be of the form, STRATEGY:VALUE, but was not", str),
		}
	}

	strategy := tokens[0]
	value := tokens[1]

	switch strategy {
	case firstPagePrefixEventID:
		return &firstPageOption{
			eventID: value,
		}, nil

	case firstPagePrefixEndsAfter:
		t, err := time.Parse(time.RFC3339, value)
		if err != nil {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  errors.Errorf("firstPageOption ends_after time \"%s\" could not be parsed", value),
			}
		}
		return &firstPageOption{
			endsAfter: &t,
		}, nil

	default:
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("firstPageOption prefix \"%s\" is not supported", strategy),
		}
	}
}

func (s *HTTPServer) getEventIDsByFilterLegacy(ctx context.Context, requestParams *getEIDsFilterOpts) (interface{}, error) {
	bcastFilter := convertFilterOptsToBroadcastFilter(requestParams)
	var eventIDs *db.EventIDs
	var err error

	if requestParams.sortBy == "hype" {
		eventIDs, err = s.OracleDB.GetEventIDsSortedByHype(ctx, bcastFilter, requestParams.desc, requestParams.cursor, requestParams.limit)
	} else {
		eventIDs, err = s.OracleDB.GetEventIDsSortedByStartTime(ctx, bcastFilter, requestParams.desc, requestParams.cursor, requestParams.limit)
	}
	if err != nil {
		return nil, err
	}

	eventIDItemModels := make([]models.V1EventIDItem, len(eventIDs.EventIDs))
	for i, eventID := range eventIDs.EventIDs {
		eventIDItemModels[i] = models.V1EventIDItem{
			EventID: eventID,
		}
	}
	if numIDs := len(eventIDs.EventIDs); numIDs > 0 {
		eventIDItemModels[numIDs-1].Cursor = eventIDs.Cursor
	}

	cacheTime := 2 * time.Minute
	if requestParams.noCache {
		cacheTime = 0
	}
	return &cachedResponse{
		payload: &models.V1EventIDsFromFilterOpts{
			Items:        eventIDItemModels,
			EventIDs:     eventIDs.EventIDs,
			LastIDCursor: eventIDs.Cursor,
			HasNextPage:  eventIDs.Cursor != "",
		},
		cacheTime: cacheTime,
	}, nil
}

func (s *HTTPServer) getEventIDsOrderedByAscStartTime(ctx context.Context, requestParams *getEIDsFilterOpts) (interface{}, error) {
	bcastFilter := convertFilterOptsToBroadcastFilter(requestParams)
	offsets := s.getEventIDsSortedByStartTimeOffsets(ctx, requestParams)

	var offset *db.EventIDsOrderedByAscTimeOffset
	var eventIDItems []db.EventIDItem
	var hasMore bool
	for _, currentOffset := range offsets {
		// Get the event IDs at the current offset.  If there aren't events at the given offsets, try the next offset.
		offset = currentOffset
		var err error

		// Work around a bug where GetEventIDsOrderedByAscStartTime can filter away event IDs it should when
		// trying to fulfill an EndsAfter offset.  Do this by converting the EndsAfter offset into an AtEventID
		// offset.
		if offset.EndsAfter != nil {
			// Get the first event that ends after the given time.
			eventIDItems, hasMore, err = s.OracleDB.GetEventIDsOrderedByAscStartTime(ctx, bcastFilter, offset, 1)
			if err != nil {
				return nil, err
			} else if len(eventIDItems) == 0 || eventIDItems[0].ID == "" {
				// If we don't find an event, move on to the next offset.
				continue
			}

			// Use the first event to to create an AtEventID offset.
			offset = &db.EventIDsOrderedByAscTimeOffset{
				AtEventID: eventIDItems[0].ID,
			}
		}

		eventIDItems, hasMore, err = s.OracleDB.GetEventIDsOrderedByAscStartTime(ctx, bcastFilter, offset, requestParams.limit)
		if err != nil {
			return nil, err
		} else if s.isOffsetFulfilled(offset, eventIDItems) {
			break
		}
	}

	eventIDs := make([]string, len(eventIDItems))
	eventIDItemModels := make([]models.V1EventIDItem, len(eventIDItems))
	for i, item := range eventIDItems {
		eventIDs[i] = item.ID
		eventIDItemModels[i] = models.V1EventIDItem{
			EventID: item.ID,
			Cursor:  item.Cursor,
		}
	}

	firstIDCursor := ""
	lastIDCursor := ""
	if len(eventIDItems) > 0 {
		firstIDCursor = eventIDItems[0].Cursor
		lastIDCursor = eventIDItems[len(eventIDItems)-1].Cursor
	}

	// Check if there are events in the opposite direction.
	// (i.e. If the request was for events at a particular place or after a cursor, is there a previous page?
	//       If the request was for events before a cursor, is there a next page?)
	invertedOffset := db.InvertEventIDsOrderedByAscTimeOffset(offset)
	hasMoreInOppositeDirection, err := s.OracleDB.HasEventIDsOrderedByAscStartTime(ctx, bcastFilter, invertedOffset)
	if err != nil {
		return nil, err
	}

	hasNextPage := hasMore
	hasPreviousPage := hasMoreInOppositeDirection
	if requestParams.getPreviousPage {
		hasNextPage = hasMoreInOppositeDirection
		hasPreviousPage = hasMore
	}

	cacheTime := 2 * time.Minute
	if requestParams.noCache {
		cacheTime = 0
	}

	return &cachedResponse{
		payload: &models.V1EventIDsFromFilterOpts{
			Items:           eventIDItemModels,
			HasPreviousPage: hasPreviousPage,
			HasNextPage:     hasNextPage,

			EventIDs:      eventIDs,
			FirstIDCursor: firstIDCursor,
			LastIDCursor:  lastIDCursor,
		},
		cacheTime: cacheTime,
	}, nil
}

func convertFilterOptsToBroadcastFilter(filterOpts *getEIDsFilterOpts) *db.BroadcastFilter {
	return &db.BroadcastFilter{
		ChannelIDs:     filterOpts.channelIDs,
		GameIDs:        filterOpts.gameIDs,
		OwnerIDs:       filterOpts.ownerIDs,
		Types:          filterOpts.types,
		ParentEventIDs: filterOpts.parentEventIDs,
		StartTimeWindow: &db.TimeWindow{
			Start: filterOpts.startTimeAfter,
			End:   filterOpts.startTimeBefore,
		},
		EndTimeWindow: &db.TimeWindow{
			Start: filterOpts.endTimeAfter,
			End:   filterOpts.endTimeBefore,
		},
	}
}

func (s *HTTPServer) getEventIDsSortedByStartTimeOffsets(ctx context.Context, requestParams *getEIDsFilterOpts) []*db.EventIDsOrderedByAscTimeOffset {
	// If a cursor is given, it means we're either getting the next/previous page after/before the event
	// that the cursor points to.  In this case, we set up the offsets using the cursor, and ignore any
	// first page options that were given.
	if cursor := requestParams.cursor; cursor != "" {
		beforeCursor := ""
		afterCursor := ""
		if requestParams.getPreviousPage {
			beforeCursor = cursor
		} else {
			afterCursor = cursor
		}

		return []*db.EventIDsOrderedByAscTimeOffset{
			{
				BeforeCursor: beforeCursor,
				AfterCursor:  afterCursor,
			},
		}
	}

	// Set up offsets using the given first page options.
	offsets := make([]*db.EventIDsOrderedByAscTimeOffset, 0)
	for _, firstPageOption := range requestParams.firstPageOptions {
		var offsetOptions *db.EventIDsOrderedByAscTimeOffset

		if firstPageOption.eventID != "" {
			offsetOptions = &db.EventIDsOrderedByAscTimeOffset{
				AtEventID: firstPageOption.eventID,
			}
		} else if firstPageOption.endsAfter != nil {
			offsetOptions = &db.EventIDsOrderedByAscTimeOffset{
				EndsAfter: firstPageOption.endsAfter,
			}
		}

		if offsetOptions == nil {
			continue
		}

		offsets = append(offsets, offsetOptions)
	}

	// Add a fallback, empty offset which will be used if the given first page options can't be fulfilled.
	offsets = append(offsets, &db.EventIDsOrderedByAscTimeOffset{})
	return offsets
}

func (s *HTTPServer) getManagedCollectionIDsByOwner(r *http.Request) (interface{}, error) {
	ctx := r.Context()
	ownerID, err := requireString(r, "owner_id")
	if err != nil {
		return nil, err
	}
	userID, err := requireString(r, "user_id")
	if err != nil {
		return nil, err
	}

	if err = s.canViewManagedEvents(ctx, ownerID, userID); err != nil {
		return nil, err
	}

	cursor := r.URL.Query().Get("cursor")
	desc, err := requestBool(r, "desc", false)
	if err != nil {
		return nil, err
	}

	limit, err := requestInt(r, "limit", 200)
	if err != nil {
		return nil, err
	}
	if limit > 200 {
		limit = 200
	}

	eventIDs, err := s.OracleDB.GetCollectionIDsByOwner(ctx, ownerID, desc, cursor, limit)
	if err != nil {
		return nil, err
	}

	return &models.V1EventIDs{
		EventIDs: eventIDs.EventIDs,
		Cursor:   eventIDs.Cursor,
	}, nil
}

func (s *HTTPServer) getManagedLeafEventIDsByOwner(r *http.Request) (interface{}, error) {
	ctx := r.Context()
	requestParams, err := s.parseGetManagedLeafEventIDsByOwnerParams(r)
	if err != nil {
		return nil, err
	}

	// Do an auth check to prevent unauthorized requests from querying the database.
	err = s.canViewManagedEvents(ctx, requestParams.ownerID, requestParams.userID)
	if err != nil {
		return nil, err
	}

	return s.getEventIDsByFilterParams(ctx, requestParams.toGetEIDsFilterOpts())
}

func (s *HTTPServer) parseGetManagedLeafEventIDsByOwnerParams(r *http.Request) (*getManagedLeafEventIDsByOwnerParams, error) {
	opts, err := s.parseGetManagedLeafEventIDsOpts(r)
	if err != nil {
		return nil, err
	}

	userID, err := requireString(r, "user_id")
	if err != nil {
		return nil, err
	}

	ownerID, err := requireString(r, "owner_id")
	if err != nil {
		return nil, err
	}

	return &getManagedLeafEventIDsByOwnerParams{
		userID:                     userID,
		ownerID:                    ownerID,
		getManagedLeafEventIDsOpts: *opts,
	}, nil
}

type getManagedLeafEventIDsByOwnerParams struct {
	ownerID string
	userID  string
	getManagedLeafEventIDsOpts
}

func (p *getManagedLeafEventIDsByOwnerParams) toGetEIDsFilterOpts() *getEIDsFilterOpts {
	opts := p.getManagedLeafEventIDsOpts.toGetEIDsFilterOpts()
	opts.sortBy = "start_time"
	opts.ownerIDs = append(opts.ownerIDs, p.ownerID)
	opts.noCache = true

	return opts
}

func (s *HTTPServer) getManagedLeafEventIDsByParent(r *http.Request) (interface{}, error) {
	ctx := r.Context()
	requestParams, err := s.parseGetManagedLeafEventIDsByParentParams(r)
	if err != nil {
		return nil, err
	}

	parentEvent, err := s.EventHandlers.GetEvent(ctx, requestParams.parentID, false, false)
	if err != nil {
		return nil, err
	} else if parentEvent == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("event not found"),
		}
	}

	// Do an auth check to prevent unauthorized requests from querying the database.
	ownerID := parentEvent.GetOwnerID()
	err = s.canViewManagedEvents(ctx, ownerID, requestParams.userID)
	if err != nil {
		return nil, err
	}

	return s.getEventIDsByFilterParams(ctx, requestParams.toGetEIDsFilterOpts())
}

func (s *HTTPServer) parseGetManagedLeafEventIDsByParentParams(r *http.Request) (*getManagedLeafEventIDsByParentParams, error) {
	opts, err := s.parseGetManagedLeafEventIDsOpts(r)
	if err != nil {
		return nil, err
	}

	userID, err := requireString(r, "user_id")
	if err != nil {
		return nil, err
	}

	parentID, err := requireString(r, "parent_id")
	if err != nil {
		return nil, err
	}

	return &getManagedLeafEventIDsByParentParams{
		userID:                     userID,
		parentID:                   parentID,
		getManagedLeafEventIDsOpts: *opts,
	}, nil
}

type getManagedLeafEventIDsByParentParams struct {
	parentID string
	userID   string
	getManagedLeafEventIDsOpts
}

func (p *getManagedLeafEventIDsByParentParams) toGetEIDsFilterOpts() *getEIDsFilterOpts {
	opts := p.getManagedLeafEventIDsOpts.toGetEIDsFilterOpts()
	opts.sortBy = "start_time"
	opts.parentEventIDs = append(opts.parentEventIDs, p.parentID)
	opts.noCache = true

	return opts
}

type getManagedLeafEventIDsOpts struct {
	gameIDs          []string
	types            []string
	startTimeAfter   *time.Time
	startTimeBefore  *time.Time
	endTimeAfter     *time.Time
	endTimeBefore    *time.Time
	limit            int
	cursor           string
	desc             bool
	getPreviousPage  bool
	firstPageOptions []firstPageOption
}

func (o *getManagedLeafEventIDsOpts) toGetEIDsFilterOpts() *getEIDsFilterOpts {
	return &getEIDsFilterOpts{
		gameIDs:          o.gameIDs,
		types:            o.types,
		startTimeAfter:   o.startTimeAfter,
		startTimeBefore:  o.startTimeBefore,
		endTimeAfter:     o.endTimeAfter,
		endTimeBefore:    o.endTimeBefore,
		limit:            o.limit,
		cursor:           o.cursor,
		desc:             o.desc,
		getPreviousPage:  o.getPreviousPage,
		firstPageOptions: o.firstPageOptions,
	}
}

func (s *HTTPServer) parseGetManagedLeafEventIDsOpts(r *http.Request) (*getManagedLeafEventIDsOpts, error) {
	gameIDs, err := requestOptionalStrings(r, "game_ids")
	if err != nil {
		return nil, err
	}

	types, err := requestOptionalStrings(r, "types")
	if err != nil {
		return nil, err
	}

	startTimeAfter, err := requestOptionalTime(r, "start_time_after")
	if err != nil {
		return nil, err
	}

	startTimeBefore, err := requestOptionalTime(r, "start_time_before")
	if err != nil {
		return nil, err
	}

	endTimeAfter, err := requestOptionalTime(r, "end_time_after")
	if err != nil {
		return nil, err
	}

	endTimeBefore, err := requestOptionalTime(r, "end_time_before")
	if err != nil {
		return nil, err
	}

	limit, err := requestInt(r, "limit", 20)
	if err != nil {
		return nil, err
	}
	if limit > 100 {
		limit = 100
	}

	cursor := r.URL.Query().Get("cursor")

	descending, err := requestBool(r, "desc", false)
	if err != nil {
		return nil, err
	}

	firstPageOptions := make([]firstPageOption, 2)
	startListAtEventID := r.URL.Query().Get("start_list_event_id")
	if startListAtEventID != "" {
		firstPageOptions = append(firstPageOptions, firstPageOption{eventID: startListAtEventID})
	}
	startListWithEndTimeAfter, err := requestOptionalTime(r, "start_list_end_time_after")
	if err != nil {
		return nil, err
	}
	if startListWithEndTimeAfter != nil {
		firstPageOptions = append(firstPageOptions, firstPageOption{endsAfter: startListWithEndTimeAfter})
	}

	getPreviousPage, err := requestBool(r, "get_prev_page", false)
	if err != nil {
		return nil, err
	}
	if getPreviousPage && cursor == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("get_prev_page was true but no cursor was given"),
		}
	}

	return &getManagedLeafEventIDsOpts{
		gameIDs:          gameIDs,
		types:            types,
		startTimeAfter:   startTimeAfter,
		startTimeBefore:  startTimeBefore,
		endTimeAfter:     endTimeAfter,
		endTimeBefore:    endTimeBefore,
		limit:            limit,
		cursor:           cursor,
		desc:             descending,
		getPreviousPage:  getPreviousPage,
		firstPageOptions: firstPageOptions,
	}, nil
}

// Check if an offset's conditions have been met.  If a condition hasn't been met, another offset can be tried.
func (s *HTTPServer) isOffsetFulfilled(offset *db.EventIDsOrderedByAscTimeOffset, eventIDItems []db.EventIDItem) bool {
	if len(eventIDItems) == 0 {
		return false
	}

	startingEventID := offset.AtEventID
	if startingEventID != "" {
		return eventIDItems[0].ID == startingEventID
	}

	return true
}

func (s *HTTPServer) reserveImageUpload(r *http.Request) (interface{}, error) {
	_, err := requireUserID(r)
	if err != nil {
		return nil, err
	}

	resp, err := s.ImageUploader.ReserveImageUpload(r.Context())
	if err != nil {
		return nil, err
	}

	return &models.ImageUploadHandle{
		UploadID:  images.GeaID(resp.UploadId),
		UploadURL: resp.Url,
	}, nil
}

func (s *HTTPServer) legacyImageUpload(r *http.Request) (interface{}, error) {
	_, err := requireUserID(r)
	if err != nil {
		return nil, err
	}

	var reqBody models.ImageUploadRequest
	err = json.NewDecoder(r.Body).Decode(&reqBody)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "unable to parse request body"),
		}
	}

	imageData, err := base64.StdEncoding.DecodeString(reqBody.Base64EncodedImage)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "unable to parse image data"),
		}
	}

	contentType := http.DetectContentType(imageData)
	if contentType != "image/jpeg" {
		return nil, &service_common.CodedError{
			Code: http.StatusUnprocessableEntity,
			Err:  errors.Errorf("image is not of type jpeg got type %s, %s", contentType, reqBody.Base64EncodedImage),
		}
	}

	resp, err := s.ImageUploader.ReserveImageUpload(r.Context())
	if err != nil {
		return nil, err
	}

	err = s.ImageUploader.SyncUploader.UploadImage(r.Context(), resp.UploadId, resp.Url, bytes.NewBuffer(imageData))
	if err != nil {
		return nil, err
	}

	return &models.ImageUploadResponse{
		UploadID: images.GeaID(resp.UploadId),
	}, nil
}

func (s *HTTPServer) createNotificationJobs(r *http.Request) (interface{}, error) {
	go s.HypemanScheduler.CreateNotificationJobs(context.Background())
	return nil, nil
}

func (s *HTTPServer) processNotificationJob(r *http.Request) (interface{}, error) {
	eventID, err := requireString(r, "event_id")
	if err != nil {
		return nil, err
	}

	err = s.HypemanWorker.ProcessNotificationJob(context.Background(), &hypeman.NotificationJob{
		EventID: eventID,
	})

	return nil, err
}

func (s *HTTPServer) processHardDeleteUserJob(r *http.Request) (interface{}, error) {
	userID, err := requireString(r, "user_id")
	if err != nil {
		return nil, err
	}

	err = s.UserHardDeleteWorker.ProcessHardDeleteUserJob(context.Background(), &harddelete.Job{
		UserID: userID,
	})

	return nil, err
}

func (s *HTTPServer) sendEventStartedNotifications(r *http.Request) (interface{}, error) {
	eventIDs, err := requireStrings(r, "ids")
	if err != nil {
		return nil, err
	}

	err = s.HypemanScheduler.EnqueueNotificationJobs(context.Background(), eventIDs)
	return nil, err
}

// publishChannelEvent facilitates testing the ChannelEventPublisher.  It allows the integration test to invoke
// the PublishChannelEvent logic using an HTTP request, instead of having ChannelEventPublisher read from an
// SQS queue.
func (s *HTTPServer) publishChannelEvent(r *http.Request) (interface{}, error) {
	channelID, err := requireString(r, "channel_id")
	if err != nil {
		return nil, err
	}

	err = s.ChannelEventPublisher.PublishChannelEvent(context.Background(), &channelevent.SQSMessage{
		ChannelID: channelID,
		Version:   channelevent.SQSVersion,
	})

	return nil, err
}

func (s *HTTPServer) CheckAWS(r *http.Request) error {
	return nil
}

func (s *HTTPServer) createChannelEventUpdates(r *http.Request) (interface{}, error) {
	go s.ChannelEventScheduler.CreateChannelEventUpdates(context.Background())
	return nil, nil
}
